const mysql = require('mysql2');
require('dotenv').config();

// Debug: Verificar que las variables de entorno se están leyendo
console.log('🔍 Configuración de la base de datos:');
const dbConfig = {
  host: 'localhost',
  user: 'root',
  password: '',
  database: 'sev_adm',
  port: 3306,
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0,
  charset: 'utf8mb4',
  typeCast: function (field, next) {
    if (field.type === 'ENUM') {
      return field.string();
    }
    return next();
  }
};

// Sobrescribir con variables de entorno si existen
if (process.env.DB_HOST) dbConfig.host = process.env.DB_HOST;
if (process.env.DB_USER) dbConfig.user = process.env.DB_USER;
if (process.env.DB_PASSWORD) dbConfig.password = process.env.DB_PASSWORD;
if (process.env.DB_NAME) dbConfig.database = process.env.DB_NAME;
if (process.env.DB_PORT) dbConfig.port = parseInt(process.env.DB_PORT);

console.log(`   DB_NAME: ${dbConfig.database}`);
console.log(`   DB_HOST: ${dbConfig.host}`);
console.log(`   DB_USER: ${dbConfig.user}`);

const pool = mysql.createPool(dbConfig);

// Probar la conexión al inicializar
pool.getConnection((err, connection) => {
  if (err) {
    console.error('❌ Error conectando a la base de datos:', err.code);
    console.log('💡 Posibles soluciones:');
    console.log('   1. Verificar que MySQL/XAMPP esté ejecutándose');
    console.log('   2. Verificar credenciales en el archivo .env');
    console.log('   3. Verificar que la base de datos exista');
    console.log(`   4. Configuración actual: ${dbConfig.user}@${dbConfig.host}:${dbConfig.port}/${dbConfig.database}`);
  } else {
    console.log(`✅ Conectado a la base de datos: ${dbConfig.database}`);
    connection.release();
  }
});

// Helper functions para operaciones comunes
const helpers = {
  async beginTransaction() {
    const connection = await pool.promise().getConnection();
    await connection.beginTransaction();
    return connection;
  },

  async commitTransaction(connection) {
    await connection.commit();
    connection.release();
  },

  async rollbackTransaction(connection) {
    await connection.rollback();
    connection.release();
  },

  async query(sql, params = []) {
    const [rows] = await pool.promise().execute(sql, params);
    return rows;
  },

  // Verifica si un registro existe
  async exists(table, field, value) {
    const [rows] = await pool.promise().execute(
      `SELECT 1 FROM ${table} WHERE ${field} = ? LIMIT 1`,
      [value]
    );
    return rows.length > 0;
  }
};

module.exports = { pool, helpers };