const jwt = require('jsonwebtoken');
const bcrypt = require('bcrypt');
// Configuración JWT
const JWT_SECRET = process.env.JWT_SECRET || 'your-secret-key-change-in-production';
const JWT_EXPIRES_IN = process.env.JWT_EXPIRES_IN || '24h';

// Rate limiting
let requestLimiter;

// Middleware para validar JWT
const authenticateToken = (req, res, next) => {
  try {
    console.log('🔑 Verificando token...');
    const authHeader = req.headers['authorization'];
    console.log('📨 Headers de autorización:', authHeader);
    
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      console.log('❌ Token no proporcionado o formato inválido');
      return res.status(401).json({ 
        success: false, 
        error: 'Token de acceso requerido' 
      });
    }

    const token = authHeader.split(' ')[1];
    console.log('🎟️ Token extraído');

    jwt.verify(token, JWT_SECRET, (err, decoded) => {
      if (err) {
        console.log('❌ Error verificando token:', err.message);
        return res.status(403).json({ 
          success: false, 
          error: 'Token inválido o expirado',
          details: process.env.NODE_ENV === 'development' ? err.message : undefined
        });
      }

      // Verificar campos requeridos en el token
      if (!decoded.userId || !decoded.id_empresa) {
        console.log('❌ Token no contiene información requerida');
        return res.status(403).json({ 
          success: false, 
          error: 'Token inválido - falta información requerida' 
        });
      }

      // Verificar estado activo
      if (!decoded.activo || !decoded.empresa_activa) {
        console.log('❌ Usuario o empresa inactivos');
        return res.status(403).json({ 
          success: false, 
          error: 'Usuario o empresa inactivos' 
        });
      }

      console.log('✅ Token verificado para usuario:', decoded.correo);
      req.user = decoded;
      next();
    });
  } catch (error) {
    console.error('❌ Error en autenticación:', error);
    return res.status(500).json({ 
      success: false, 
      error: 'Error de autenticación',
      details: process.env.NODE_ENV === 'development' ? error.message : undefined
    });
  }
};

// Middleware para validar roles
const authorizeRole = (roles) => {
  return (req, res, next) => {
    if (!roles.includes(req.user.rol)) {
      return res.status(403).json({ 
        success: false, 
        error: 'No tienes permisos para realizar esta acción' 
      });
    }
    next();
  };
};

// Función para generar token JWT
const generateToken = (user) => {
  return jwt.sign(
    { 
      id: user.id, 
      correo: user.correo, 
      rol: user.rol, 
      id_empresa: user.id_empresa 
    },
    JWT_SECRET,
    { expiresIn: JWT_EXPIRES_IN }
  );
};

// Función para hashear contraseña
const hashPassword = async (password) => {
  const saltRounds = 12;
  return await bcrypt.hash(password, saltRounds);
};

// Función para verificar contraseña
const verifyPassword = async (password, hashedPassword) => {
  return await bcrypt.compare(password, hashedPassword);
};

// Validaciones de entrada
const validateEmail = (email) => {
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  return emailRegex.test(email);
};

const validatePassword = (password) => {
  // Mínimo 8 caracteres, al menos una letra mayúscula, una minúscula y un número
  const passwordRegex = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)[a-zA-Z\d@$!%*?&]{8,}$/;
  return passwordRegex.test(password);
};

const validatePhone = (phone) => {
  const phoneRegex = /^\+?[\d\s\-\(\)]{8,15}$/;
  return phoneRegex.test(phone);
};

// Función para sanitizar entrada SQL
const sanitizeInput = (input) => {
  if (typeof input !== 'string') return input;
  return input.trim().replace(/[<>\"'%;()&+]/g, '');
};

// Middleware de validación de entrada
const validateInput = (validations) => {
  return (req, res, next) => {
    const errors = [];

    for (const [field, rules] of Object.entries(validations)) {
      const value = req.body[field];

      if (rules.required && (!value || value.toString().trim() === '')) {
        errors.push(`${field} es requerido`);
        continue;
      }

      if (value) {
        if (rules.type === 'email' && !validateEmail(value)) {
          errors.push(`${field} debe ser un email válido`);
        }

        if (rules.type === 'password' && !validatePassword(value)) {
          errors.push(`${field} debe tener al menos 8 caracteres, una mayúscula, una minúscula y un número`);
        }

        if (rules.type === 'phone' && !validatePhone(value)) {
          errors.push(`${field} debe ser un número de teléfono válido`);
        }

        if (rules.minLength && value.length < rules.minLength) {
          errors.push(`${field} debe tener al menos ${rules.minLength} caracteres`);
        }

        if (rules.maxLength && value.length > rules.maxLength) {
          errors.push(`${field} no puede tener más de ${rules.maxLength} caracteres`);
        }

        if (rules.min && parseInt(value) < rules.min) {
          errors.push(`${field} debe ser mayor a ${rules.min}`);
        }

        if (rules.max && parseInt(value) > rules.max) {
          errors.push(`${field} debe ser menor a ${rules.max}`);
        }
      }
    }

    if (errors.length > 0) {
      return res.status(400).json({
        success: false,
        error: 'Errores de validación',
        details: errors
      });
    }

    // Sanitizar inputs
    for (const field in req.body) {
      req.body[field] = sanitizeInput(req.body[field]);
    }

    next();
  };
};

// Middleware de rate limiting simple
const rateLimitMap = new Map();

const rateLimit = (windowMs, maxRequests) => {
  return (req, res, next) => {
    const clientIP = req.ip || req.connection.remoteAddress;
    const now = Date.now();
    const windowStart = now - windowMs;

    if (!rateLimitMap.has(clientIP)) {
      rateLimitMap.set(clientIP, []);
    }

    const requests = rateLimitMap.get(clientIP);
    
    // Limpiar requests antiguos
    const validRequests = requests.filter(time => time > windowStart);
    rateLimitMap.set(clientIP, validRequests);

    if (validRequests.length >= maxRequests) {
      return res.status(429).json({
        success: false,
        error: 'Demasiadas solicitudes. Intenta más tarde.'
      });
    }

    validRequests.push(now);
    rateLimitMap.set(clientIP, validRequests);

    next();
  };
};

// Función para manejar errores de base de datos
const handleDatabaseError = (error, res) => {
  console.error('Error de base de datos:', error);
  
  if (error.code === 'ER_DUP_ENTRY') {
    return res.status(409).json({
      success: false,
      error: 'El registro ya existe'
    });
  }
  
  if (error.code === 'ER_NO_REFERENCED_ROW_2') {
    return res.status(400).json({
      success: false,
      error: 'Referencia inválida'
    });
  }
  
  return res.status(500).json({
    success: false,
    error: 'Error interno del servidor'
  });
};

// Logging middleware
function logRequest(req, res, next) {
  const timestamp = new Date().toISOString();
  console.log(`${timestamp} - ${req.method} ${req.url}`);
  next();
}

// Exportar módulos
module.exports = {
  authenticateToken,
  authorizeRole,
  generateToken,
  hashPassword,
  verifyPassword,
  validateEmail,
  validatePassword,
  validatePhone,
  sanitizeInput,
  validateInput,
  rateLimit,
  logRequest,
  handleDatabaseError,
  JWT_SECRET
};
