// Rutas para manejo de trabajadores
const express = require('express');
const bcrypt = require('bcrypt');
const { pool, helpers } = require('./db');
const router = express.Router();

// Middleware para agregar la conexión a la base de datos al request
router.use((req, res, next) => {
  req.db = pool.promise();
  next();
});

// Middleware para verificar que el usuario sea administrador
const verificarAdmin = (req, res, next) => {
  // Por ahora, permitir acceso sin verificación estricta del rol
  // En una implementación real, verificarías el token JWT
  console.log('🔐 Verificando permisos de administrador...');
  
  // Verificar que hay un token de Authorization
  const authHeader = req.headers['authorization'];
  if (!authHeader || !authHeader.startsWith('Bearer ')) {
    return res.status(401).json({ 
      success: false, 
      message: 'Token de autorización requerido' 
    });
  }
  
  // Por ahora, asumir que cualquier usuario con token válido puede acceder
  // TODO: Implementar verificación de rol real
  next();
};

// Obtener todos los trabajadores de una empresa (usando solo tabla usuarios)
router.get('/empresa/:id_empresa', verificarAdmin, async (req, res) => {
  const { id_empresa } = req.params;
  
  try {
    console.log(`🔍 Obteniendo trabajadores para empresa ${id_empresa}...`);
    
    // Query simplificado usando solo la tabla usuarios con campos que existen
    const query = `
      SELECT 
        u.id,
        u.id_empresa,
        u.nombre,
        u.correo,
        u.rol,
        u.activo,
        e.nombre as empresa_nombre
      FROM usuarios u
      LEFT JOIN empresas e ON u.id_empresa = e.id
      WHERE u.id_empresa = ?
      ORDER BY u.id DESC
    `;
    
    const [trabajadores] = await req.db.execute(query, [id_empresa]);
    
    console.log(`✅ Encontrados ${trabajadores.length} trabajadores para empresa ${id_empresa}`);
    
    res.json({
      success: true,
      trabajadores: trabajadores,
      mensaje: `${trabajadores.length} trabajadores encontrados`
    });
  } catch (error) {
    console.error('❌ Error al obtener trabajadores:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor: ' + error.message
    });
  }
});

// Crear nuevo trabajador/usuario
router.post('/', verificarAdmin, async (req, res) => {
  const {
    nombre,
    correo,
    clave,
    rol = 'VENDEDOR',
    id_empresa
  } = req.body;

  // Validaciones
  if (!nombre || !correo || !clave || !id_empresa) {
    return res.status(400).json({
      success: false,
      message: 'Faltan campos obligatorios: nombre, correo, contraseña y empresa'
    });
  }

  if (clave.length < 8) {
    return res.status(400).json({
      success: false,
      message: 'La contraseña debe tener al menos 8 caracteres'
    });
  }

  // Validar rol
  const rolesValidos = ['ADMIN', 'VENDEDOR', 'SUPERVISOR', 'MECANICO'];
  if (!rolesValidos.includes(rol)) {
    return res.status(400).json({
      success: false,
      message: 'Rol inválido. Debe ser: ' + rolesValidos.join(', ')
    });
  }

  try {
    console.log(`➕ Creando nuevo usuario: ${nombre} (${correo}) con rol ${rol}...`);
    
    // Verificar que el correo no esté en uso
    const [existeUsuario] = await req.db.execute(
      'SELECT id FROM usuarios WHERE correo = ?',
      [correo]
    );

    if (existeUsuario.length > 0) {
      return res.status(409).json({
        success: false,
        message: 'Ya existe un usuario con ese correo electrónico'
      });
    }

    // Verificar que la empresa existe
    const [empresaExiste] = await req.db.execute(
      'SELECT id FROM empresas WHERE id = ?',
      [id_empresa]
    );

    if (empresaExiste.length === 0) {
      return res.status(400).json({
        success: false,
        message: 'La empresa especificada no existe'
      });
    }

    // Hashear contraseña
    const saltRounds = 10;
    const hashedPassword = await bcrypt.hash(clave, saltRounds);

    // Crear usuario
    const [result] = await req.db.execute(
      `INSERT INTO usuarios (id_empresa, nombre, correo, clave, rol, activo) 
       VALUES (?, ?, ?, ?, ?, TRUE)`,
      [id_empresa, nombre, correo, hashedPassword, rol]
    );

    const nuevoUsuarioId = result.insertId;

    console.log(`✅ Usuario creado exitosamente con ID: ${nuevoUsuarioId}`);

    res.status(201).json({
      success: true,
      message: 'Trabajador creado exitosamente',
      usuario: {
        id: nuevoUsuarioId,
        nombre,
        correo,
        rol,
        id_empresa,
        activo: true
      }
    });

  } catch (error) {
    console.error('❌ Error al crear trabajador:', error);
    
    if (error.code === 'ER_DUP_ENTRY') {
      res.status(409).json({
        success: false,
        message: 'Ya existe un usuario con ese correo electrónico'
      });
    } else {
      res.status(500).json({
        success: false,
        message: 'Error interno del servidor: ' + error.message
      });
    }
  }
});

// TODO: Estos endpoints requieren la tabla trabajadores que no existe aún
/*
// Obtener un trabajador específico
router.get('/:id', verificarAdmin, async (req, res) => {
  const { id } = req.params;
  
  try {
    const query = `
      SELECT 
        t.*,
        u.nombre,
        u.correo,
        u.rol,
        u.activo as usuario_activo
      FROM trabajadores t
      JOIN usuarios u ON t.id_usuario = u.id
      WHERE t.id = ?
    `;
    
    const [result] = await req.db.execute(query, [id]);
    
    if (result.length === 0) {
      return res.status(404).json({
        success: false,
        message: 'Trabajador no encontrado'
      });
    }
    
    res.json({
      success: true,
      trabajador: result[0]
    });
  } catch (error) {
    console.error('Error al obtener trabajador:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Crear nuevo trabajador
router.post('/', verificarAdmin, async (req, res) => {
  const {
    id_empresa,
    nombre,
    correo,
    clave,
    rol = 'trabajador',
    cargo,
    telefono_trabajo,
    fecha_ingreso,
    descuento_porcentaje = 0,
    descuento_monto_fijo = 0,
    tipo_descuento_principal = 'porcentaje',
    comision_porcentaje = 0,
    salario_base = 0,
    observaciones,
    puede_hacer_descuentos = false,
    puede_ver_precios_compra = false,
    puede_ver_ganancias = false,
    puede_acceder_reportes = false
  } = req.body;

  // Validaciones
  if (!nombre || !correo || !clave || !cargo) {
    return res.status(400).json({
      success: false,
      message: 'Faltan campos obligatorios: nombre, correo, contraseña y cargo'
    });
  }

  if (clave.length < 8) {
    return res.status(400).json({
      success: false,
      message: 'La contraseña debe tener al menos 8 caracteres'
    });
  }

  try {
    // Verificar que el correo no esté en uso
    const [existeUsuario] = await req.db.execute(
      'SELECT id FROM usuarios WHERE correo = ?',
      [correo]
    );

    if (existeUsuario.length > 0) {
      return res.status(409).json({
        success: false,
        message: 'Ya existe un usuario con ese correo electrónico'
      });
    }

    // Iniciar transacción
    await req.db.execute('START TRANSACTION');

    try {
      // Hashear contraseña
      const saltRounds = 10;
      const hashedPassword = await bcrypt.hash(clave, saltRounds);

      // Crear usuario
      const [resultUsuario] = await req.db.execute(
        `INSERT INTO usuarios (id_empresa, nombre, correo, clave, rol, activo) 
         VALUES (?, ?, ?, ?, ?, TRUE)`,
        [id_empresa, nombre, correo, hashedPassword, rol]
      );

      const idUsuario = resultUsuario.insertId;

      // Crear trabajador
      const [resultTrabajador] = await req.db.execute(
        `INSERT INTO trabajadores 
         (id_usuario, id_empresa, cargo, descuento_porcentaje, descuento_monto_fijo, 
          tipo_descuento_principal, comision_porcentaje, salario_base, telefono_trabajo, 
          fecha_ingreso, observaciones, puede_hacer_descuentos, puede_ver_precios_compra, 
          puede_ver_ganancias, puede_acceder_reportes, activo) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, TRUE)`,
        [
          idUsuario, id_empresa, cargo, descuento_porcentaje, descuento_monto_fijo,
          tipo_descuento_principal, comision_porcentaje, salario_base, telefono_trabajo,
          fecha_ingreso, observaciones, puede_hacer_descuentos, puede_ver_precios_compra,
          puede_ver_ganancias, puede_acceder_reportes
        ]
      );

      // Confirmar transacción
      await req.db.execute('COMMIT');

      res.status(201).json({
        success: true,
        message: 'Trabajador creado exitosamente',
        trabajador_id: resultTrabajador.insertId,
        usuario_id: idUsuario
      });

    } catch (error) {
      // Revertir transacción en caso de error
      await req.db.execute('ROLLBACK');
      throw error;
    }

  } catch (error) {
    console.error('Error al crear trabajador:', error);
    
    if (error.code === 'ER_DUP_ENTRY') {
      res.status(409).json({
        success: false,
        message: 'Ya existe un usuario con ese correo electrónico'
      });
    } else {
      res.status(500).json({
        success: false,
        message: 'Error interno del servidor'
      });
    }
  }
});

// Actualizar trabajador
router.put('/:id', verificarAdmin, async (req, res) => {
  const { id } = req.params;
  const {
    nombre,
    correo,
    clave,
    rol,
    cargo,
    telefono_trabajo,
    fecha_ingreso,
    descuento_porcentaje,
    descuento_monto_fijo,
    tipo_descuento_principal,
    comision_porcentaje,
    salario_base,
    observaciones,
    puede_hacer_descuentos,
    puede_ver_precios_compra,
    puede_ver_ganancias,
    puede_acceder_reportes
  } = req.body;

  try {
    // Obtener información actual del trabajador
    const [trabajadorActual] = await req.db.execute(
      'SELECT id_usuario FROM trabajadores WHERE id = ?',
      [id]
    );

    if (trabajadorActual.length === 0) {
      return res.status(404).json({
        success: false,
        message: 'Trabajador no encontrado'
      });
    }

    const idUsuario = trabajadorActual[0].id_usuario;

    // Iniciar transacción
    await req.db.execute('START TRANSACTION');

    try {
      // Preparar query de actualización para usuario
      let updateUsuarioQuery = 'UPDATE usuarios SET nombre = ?, correo = ?, rol = ?';
      let updateUsuarioParams = [nombre, correo, rol];

      // Si se proporciona nueva contraseña, incluirla
      if (clave && clave.length >= 8) {
        const saltRounds = 10;
        const hashedPassword = await bcrypt.hash(clave, saltRounds);
        updateUsuarioQuery += ', clave = ?';
        updateUsuarioParams.push(hashedPassword);
      }

      updateUsuarioQuery += ' WHERE id = ?';
      updateUsuarioParams.push(idUsuario);

      // Actualizar usuario
      await req.db.execute(updateUsuarioQuery, updateUsuarioParams);

      // Actualizar trabajador
      await req.db.execute(
        `UPDATE trabajadores SET 
         cargo = ?, descuento_porcentaje = ?, descuento_monto_fijo = ?, 
         tipo_descuento_principal = ?, comision_porcentaje = ?, salario_base = ?, 
         telefono_trabajo = ?, fecha_ingreso = ?, observaciones = ?, 
         puede_hacer_descuentos = ?, puede_ver_precios_compra = ?, 
         puede_ver_ganancias = ?, puede_acceder_reportes = ?,
         fecha_actualizacion = CURRENT_TIMESTAMP
         WHERE id = ?`,
        [
          cargo, descuento_porcentaje, descuento_monto_fijo,
          tipo_descuento_principal, comision_porcentaje, salario_base,
          telefono_trabajo, fecha_ingreso, observaciones,
          puede_hacer_descuentos, puede_ver_precios_compra,
          puede_ver_ganancias, puede_acceder_reportes, id
        ]
      );

      // Confirmar transacción
      await req.db.execute('COMMIT');

      res.json({
        success: true,
        message: 'Trabajador actualizado exitosamente'
      });

    } catch (error) {
      // Revertir transacción en caso de error
      await req.db.execute('ROLLBACK');
      throw error;
    }

  } catch (error) {
    console.error('Error al actualizar trabajador:', error);
    
    if (error.code === 'ER_DUP_ENTRY') {
      res.status(409).json({
        success: false,
        message: 'Ya existe un usuario con ese correo electrónico'
      });
    } else {
      res.status(500).json({
        success: false,
        message: 'Error interno del servidor'
      });
    }
  }
});

// Cambiar estado de un trabajador (activar/desactivar) usando solo tabla usuarios
router.put('/:id/estado', verificarAdmin, async (req, res) => {
  const { id } = req.params;
  const { activo } = req.body;

  try {
    console.log(`🔄 Cambiando estado del usuario ${id} a ${activo ? 'activo' : 'inactivo'}...`);
    
    // Verificar que el usuario existe
    const [usuario] = await req.db.execute(
      'SELECT id, nombre FROM usuarios WHERE id = ?',
      [id]
    );

    if (usuario.length === 0) {
      return res.status(404).json({
        success: false,
        message: 'Usuario no encontrado'
      });
    }

    // Actualizar estado del usuario (sin fecha_actualizacion que no existe)
    await req.db.execute(
      'UPDATE usuarios SET activo = ? WHERE id = ?',
      [activo, id]
    );

    console.log(`✅ Estado del usuario ${usuario[0].nombre} actualizado exitosamente`);

    res.json({
      success: true,
      message: `Usuario ${activo ? 'activado' : 'desactivado'} exitosamente`
    });

  } catch (error) {
    console.error('❌ Error al cambiar estado del usuario:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor: ' + error.message
    });
  }
});

// Obtener estadísticas de trabajadores
router.get('/stats/empresa/:id_empresa', verificarAdmin, async (req, res) => {
  const { id_empresa } = req.params;

  try {
    // Contar trabajadores por estado
    const [stats] = await req.db.execute(`
      SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN t.activo = TRUE AND u.activo = TRUE THEN 1 ELSE 0 END) as activos,
        SUM(CASE WHEN t.activo = FALSE OR u.activo = FALSE THEN 1 ELSE 0 END) as inactivos,
        SUM(CASE WHEN t.puede_hacer_descuentos = TRUE THEN 1 ELSE 0 END) as con_descuentos,
        AVG(t.comision_porcentaje) as comision_promedio,
        COUNT(DISTINCT t.cargo) as tipos_cargo
      FROM trabajadores t
      JOIN usuarios u ON t.id_usuario = u.id
      WHERE t.id_empresa = ?
    `, [id_empresa]);

    // Contar por cargo
    const [porCargo] = await req.db.execute(`
      SELECT 
        t.cargo,
        COUNT(*) as cantidad
      FROM trabajadores t
      JOIN usuarios u ON t.id_usuario = u.id
      WHERE t.id_empresa = ? AND t.activo = TRUE AND u.activo = TRUE
      GROUP BY t.cargo
      ORDER BY cantidad DESC
    `, [id_empresa]);

    res.json({
      success: true,
      estadisticas: stats[0],
      por_cargo: porCargo
    });

  } catch (error) {
    console.error('Error al obtener estadísticas:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});
*/

module.exports = router;
