const express = require('express');
const middleware = require('./middleware');
const { pool, helpers } = require('./db');
const multer = require('multer');
const debug = require('debug')('app:routes');
const path = require('path');
const fs = require('fs');
const bcrypt = require('bcrypt');
const jwt = require('jsonwebtoken');
const crypto = require('crypto');

// Configurar multer para guardar imágenes organizadas por mes
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const currentDate = new Date();
    const year = currentDate.getFullYear();
    const month = String(currentDate.getMonth() + 1).padStart(2, '0'); // Mes con 2 dígitos (01-12)
    
    // Crear estructura: uploads/motos/YYYY/MM/
    const uploadDir = path.join(__dirname, 'uploads', 'motos', year.toString(), month);
    
    if (!fs.existsSync(uploadDir)){
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: function (req, file, cb) {
    // Mantener timestamp único para evitar conflictos
    cb(null, Date.now() + '-' + file.originalname);
  }
});

// Configurar multer para documentos PDF
const documentStorage = multer.diskStorage({
  destination: function (req, file, cb) {
    const year = new Date().getFullYear();
    const month = String(new Date().getMonth() + 1).padStart(2, '0');
    const uploadDir = path.join(__dirname, 'uploads', 'documentos', year.toString(), month);
    
    if (!fs.existsSync(uploadDir)){
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: function (req, file, cb) {
    const timestamp = Date.now();
    const extension = path.extname(file.originalname);
    cb(null, `${timestamp}_${file.originalname}`);
  }
});

const upload = multer({ 
  storage: storage,
  limits: { fileSize: 5 * 1024 * 1024 }, // 5MB límite
  fileFilter: (req, file, cb) => {
    if (file.mimetype.startsWith('image/')) {
      cb(null, true);
    } else {
      cb(new Error('Solo se permiten imágenes'));
    }
  }
});

const uploadDocument = multer({
  storage: documentStorage,
  limits: { fileSize: 10 * 1024 * 1024 }, // 10MB límite
  fileFilter: (req, file, cb) => {
    const allowedTypes = [
      'application/pdf',
      'application/msword',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
      'image/jpeg',
      'image/jpg', 
      'image/png'
    ];
    
    if (allowedTypes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(new Error('Tipo de archivo no permitido. Use PDF, DOC, DOCX, JPG o PNG'));
    }
  }
});

// Helper para generar hash de archivo
const generateFileHash = (filePath) => {
  return new Promise((resolve, reject) => {
    const hash = crypto.createHash('sha256');
    const stream = fs.createReadStream(filePath);
    
    stream.on('error', reject);
    stream.on('data', chunk => hash.update(chunk));
    stream.on('end', () => resolve(hash.digest('hex')));
  });
};

// Helper para guardar documento en base de datos
const guardarDocumentoBD = async (connection, documentoData) => {
  const {
    tipo_entidad,
    id_entidad,
    tipo_documento,
    nombre_archivo,
    ruta_archivo,
    tamaño_archivo,
    mime_type = 'application/pdf',
    hash_archivo,
    descripcion,
    es_publico = 0,
    creado_por
  } = documentoData;

  // Debug: verificar que todos los campos requeridos estén presentes
  console.log('📄 Datos para guardar en BD:', {
    tipo_entidad,
    id_entidad,
    tipo_documento,
    nombre_archivo,
    ruta_archivo,
    tamaño_archivo,
    mime_type,
    hash_archivo,
    descripcion,
    es_publico,
    creado_por
  });

  // Convertir undefined a null para MySQL
  const parametros = [
    tipo_entidad || null,
    id_entidad || null,
    tipo_documento || null,
    nombre_archivo || null,
    ruta_archivo || null,
    tamaño_archivo || null,
    mime_type || 'application/pdf',
    hash_archivo || null,
    descripcion || null,
    es_publico || 0,
    creado_por || null
  ];

  console.log('📄 Parámetros para query SQL:', parametros);

  const insertQuery = `
    INSERT INTO documentos (
      tipo_entidad, id_entidad, tipo_documento, nombre_archivo, 
      ruta_archivo, tamaño_archivo, mime_type, hash_archivo, 
      descripcion, es_publico, creado_por
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
  `;

  const [result] = await connection.execute(insertQuery, parametros);

  return result.insertId;
};

const router = express.Router();

const JWT_SECRET = process.env.JWT_SECRET || 'your-secret-key-change-in-production';

// Aplicar logging a todas las rutas
router.use(middleware.logRequest);

// Endpoint para listar clientes
router.get('/clientes', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa } = req.query;
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    const query = `
      SELECT 
        c.id,
        c.nombres,
        c.apellidos,
        c.cedula,
        c.tipo_cliente,
        c.id_region,
        JSON_OBJECT(
          'id', r.id,
          'nombre', r.nombre,
          'codigo', r.codigo,
          'activa', r.activa
        ) as region,
        c.edad,
        c.telefono,
        c.email,
        c.direccion,
        c.fecha_creacion,
        c.fecha_actualizacion
      FROM clientes c
      LEFT JOIN regiones r ON c.id_region = r.id
      WHERE c.id_empresa = ?
      ORDER BY c.fecha_creacion DESC
    `;
    
    const clientes = await helpers.query(query, [empresa]);
    
    res.json({ 
      success: true,
      data: clientes
    });
  } catch (error) {
    console.error('Error al obtener clientes:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para obtener total de clientes
router.get('/clientes/total', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa, mes } = req.query;
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    const empresaId = parseInt(empresa, 10);

    // Preparar la consulta según si hay filtro de mes
    let query, params;
    
    if (mes) {
      // Si se especifica un mes, contar solo los clientes de ese mes
      query = `
        SELECT COUNT(*) as total 
        FROM clientes 
        WHERE id_empresa = ? 
        AND DATE_FORMAT(fecha_creacion, '%Y-%m') = ?
      `;
      params = [empresaId, mes];
    } else {
      // Si no se especifica mes, contar todos los clientes de la empresa
      query = 'SELECT COUNT(*) as total FROM clientes WHERE id_empresa = ?';
      params = [empresaId];
    }
    
    const result = await helpers.query(query, params);
    
    // El resultado debe ser un array con un objeto que tiene la propiedad 'total'
    const total = result && result[0] && typeof result[0].total !== 'undefined' ? result[0].total : 0;
    
    res.json({ 
      success: true,
      total: total
    });
  } catch (error) {
    console.error('Error al obtener total de clientes:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para listar ventas
router.get('/ventas', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa, mes } = req.query;
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    let query = `
      SELECT 
        v.id,
        v.fecha_venta,
        v.precio_venta,
        v.descuento,
        v.comision,
        v.metodo_pago,
        CONCAT(c.nombres, ' ', c.apellidos) as cliente_nombre,
        c.cedula as cliente_cedula,
        c.tipo_cliente,
        ma.nombre as marca,
        mo.nombre as modelo,
        mo.categoria,
        m.color,
        m.año,
        m.patente,
        u.nombre as vendedor,
        u.rol as rol_vendedor,
        df.tipo_financiamiento,
        df.monto_financiado,
        df.plazo_meses,
        df.tasa_interes,
        v.mes_venta,
        v.año_venta,
        v.observaciones,
        v.fecha_creacion,
        v.fecha_actualizacion,
        uc.nombre as creado_por,
        um.nombre as modificado_por
      FROM ventas v
      LEFT JOIN clientes c ON v.id_cliente = c.id
      LEFT JOIN motos m ON v.id_moto = m.id_moto
      LEFT JOIN modelos mo ON m.id_modelo = mo.id
      LEFT JOIN marcas ma ON mo.id_marca = ma.id
      LEFT JOIN usuarios u ON v.id_vendedor = u.id
      LEFT JOIN usuarios uc ON v.creado_por = uc.id
      LEFT JOIN usuarios um ON v.modificado_por = um.id
      LEFT JOIN detalles_financiamiento df ON v.id = df.id_venta
      WHERE v.id_empresa = ?
    `;
    
    let params = [empresa];
    
    if (mes) {
      query += ' AND DATE_FORMAT(v.fecha_venta, "%Y-%m") = ?';
      params.push(mes);
    }
    
    query += ' ORDER BY v.fecha_venta DESC';
    
    const ventas = await helpers.query(query, params);
    
    res.json({ 
      success: true,
      data: ventas
    });
  } catch (error) {
    console.error('Error al obtener ventas:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para obtener total de ventas
router.get('/ventas/total', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa, mes } = req.query;
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    const empresaId = parseInt(empresa, 10);

    let query, params;
    
    if (mes) {
      // Si se especifica un mes, contar solo las ventas de ese mes
      query = `
        SELECT COUNT(*) as total 
        FROM ventas 
        WHERE id_empresa = ? 
        AND DATE_FORMAT(fecha_venta, '%Y-%m') = ?
      `;
      params = [empresaId, mes];
    } else {
      // Si no se especifica mes, contar todas las ventas de la empresa
      query = 'SELECT COUNT(*) as total FROM ventas WHERE id_empresa = ?';
      params = [empresaId];
    }
    
    const result = await helpers.query(query, params);
    
    const total = result && result[0] && typeof result[0].total !== 'undefined' ? result[0].total : 0;
    
    res.json({ 
      success: true,
      total: total
    });
  } catch (error) {
    console.error('Error al obtener total de ventas:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para obtener ventas mensuales del año actual
router.get('/ventas/mensuales', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa, año } = req.query;
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    const empresaId = parseInt(empresa, 10);
    const añoActual = año ? parseInt(año, 10) : new Date().getFullYear();

    // Consulta para obtener ventas mensuales del año especificado
    const query = `
      SELECT 
        mes_venta as mes,
        COUNT(*) as total_ventas,
        SUM(precio_venta) as monto_total,
        AVG(precio_venta) as precio_promedio
      FROM ventas 
      WHERE id_empresa = ? 
      AND año_venta = ?
      GROUP BY mes_venta
      ORDER BY mes_venta
    `;
    
    const result = await helpers.query(query, [empresaId, añoActual]);
    
    // Crear array con todos los meses (1-12) inicializados en 0
    const ventasMensuales = Array.from({ length: 12 }, (_, index) => ({
      mes: index + 1,
      total_ventas: 0,
      monto_total: 0,
      precio_promedio: 0
    }));
    
    // Llenar los datos reales de ventas
    result.forEach(venta => {
      const mesIndex = venta.mes - 1; // Convertir a índice 0-based
      if (mesIndex >= 0 && mesIndex < 12) {
        ventasMensuales[mesIndex] = {
          mes: venta.mes,
          total_ventas: parseInt(venta.total_ventas) || 0,
          monto_total: parseFloat(venta.monto_total) || 0,
          precio_promedio: parseFloat(venta.precio_promedio) || 0
        };
      }
    });
    
    res.json({ 
      success: true,
      año: añoActual,
      data: ventasMensuales
    });
  } catch (error) {
    console.error('Error al obtener ventas mensuales:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint de prueba para verificar que las rutas funcionen
router.get('/test', (req, res) => {
  res.json({ success: true, message: 'API funcionando correctamente', timestamp: new Date().toISOString() });
});

// Endpoint temporal sin autenticación para debug
router.get('/ventas/comparacion-anual-debug', async (req, res) => {
  try {
    const { empresa } = req.query;
    
    console.log('🔍 Debug - Parámetros recibidos:', { empresa });
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    const empresaId = parseInt(empresa, 10);
    const añoActual = new Date().getFullYear();
    const añoAnterior = añoActual - 1;

    console.log('🔍 Debug - Consultando años:', { añoActual, añoAnterior, empresaId });

    // Consulta simple para verificar conexión
    const testQuery = 'SELECT COUNT(*) as total FROM ventas WHERE id_empresa = ?';
    const testResult = await helpers.query(testQuery, [empresaId]);
    
    console.log('🔍 Debug - Resultado prueba:', testResult);

    // Consulta principal
    const query = `
      SELECT 
        año_venta as año,
        mes_venta as mes,
        COUNT(*) as total_ventas,
        SUM(precio_venta) as monto_total,
        AVG(precio_venta) as precio_promedio
      FROM ventas 
      WHERE id_empresa = ? 
      AND año_venta IN (?, ?)
      GROUP BY año_venta, mes_venta
      ORDER BY año_venta, mes_venta
    `;
    
    const result = await helpers.query(query, [empresaId, añoAnterior, añoActual]);
    
    console.log('🔍 Debug - Resultado consulta:', result);
    
    res.json({ 
      success: true,
      debug: true,
      parametros: { empresa, empresaId, añoActual, añoAnterior },
      testResult,
      queryResult: result,
      message: 'Consulta ejecutada correctamente'
    });
  } catch (error) {
    console.error('❌ Debug - Error:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message,
      debug: true
    });
  }
});

// Endpoint para obtener comparación de ventas entre dos años
router.get('/ventas/comparacion-anual', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa, año_actual, año_anterior } = req.query;
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    const empresaId = parseInt(empresa, 10);
    const añoActual = año_actual ? parseInt(año_actual, 10) : new Date().getFullYear();
    const añoAnterior = año_anterior ? parseInt(año_anterior, 10) : añoActual - 1;

    // Consulta para obtener ventas de ambos años agrupadas por mes
    const query = `
      SELECT 
        año_venta as año,
        mes_venta as mes,
        COUNT(*) as total_ventas,
        SUM(precio_venta) as monto_total,
        AVG(precio_venta) as precio_promedio
      FROM ventas 
      WHERE id_empresa = ? 
      AND año_venta IN (?, ?)
      GROUP BY año_venta, mes_venta
      ORDER BY año_venta, mes_venta
    `;
    
    const result = await helpers.query(query, [empresaId, añoAnterior, añoActual]);
    
    // Organizar datos por año
    const datosAñoActual = Array.from({ length: 12 }, (_, index) => ({
      mes: index + 1,
      total_ventas: 0,
      monto_total: 0,
      precio_promedio: 0
    }));
    
    const datosAñoAnterior = Array.from({ length: 12 }, (_, index) => ({
      mes: index + 1,
      total_ventas: 0,
      monto_total: 0,
      precio_promedio: 0
    }));
    
    // Llenar los datos reales
    result.forEach(venta => {
      const mesIndex = venta.mes - 1;
      if (mesIndex >= 0 && mesIndex < 12) {
        const datosVenta = {
          mes: venta.mes,
          total_ventas: parseInt(venta.total_ventas) || 0,
          monto_total: parseFloat(venta.monto_total) || 0,
          precio_promedio: parseFloat(venta.precio_promedio) || 0
        };
        
        if (venta.año === añoActual) {
          datosAñoActual[mesIndex] = datosVenta;
        } else if (venta.año === añoAnterior) {
          datosAñoAnterior[mesIndex] = datosVenta;
        }
      }
    });
    
    // Calcular totales anuales
    const totalActual = datosAñoActual.reduce((sum, mes) => sum + mes.monto_total, 0);
    const totalAnterior = datosAñoAnterior.reduce((sum, mes) => sum + mes.monto_total, 0);
    const ventasActual = datosAñoActual.reduce((sum, mes) => sum + mes.total_ventas, 0);
    const ventasAnterior = datosAñoAnterior.reduce((sum, mes) => sum + mes.total_ventas, 0);
    
    // Calcular mes actual hasta la fecha
    const mesActualNumero = new Date().getMonth() + 1;
    const totalHastaMesActual = datosAñoActual.slice(0, mesActualNumero).reduce((sum, mes) => sum + mes.monto_total, 0);
    const totalHastaMesAnterior = datosAñoAnterior.slice(0, mesActualNumero).reduce((sum, mes) => sum + mes.monto_total, 0);
    
    res.json({ 
      success: true,
      año_actual: añoActual,
      año_anterior: añoAnterior,
      mes_actual: mesActualNumero,
      datos_año_actual: datosAñoActual,
      datos_año_anterior: datosAñoAnterior,
      totales: {
        año_actual: totalActual,
        año_anterior: totalAnterior,
        ventas_año_actual: ventasActual,
        ventas_año_anterior: ventasAnterior,
        hasta_mes_actual: totalHastaMesActual,
        hasta_mes_anterior: totalHastaMesAnterior,
        diferencia_total: totalActual - totalAnterior,
        diferencia_hasta_mes: totalHastaMesActual - totalHastaMesAnterior,
        porcentaje_cambio: totalAnterior > 0 ? ((totalActual - totalAnterior) / totalAnterior) * 100 : 0,
        porcentaje_cambio_mes: totalHastaMesAnterior > 0 ? ((totalHastaMesActual - totalHastaMesAnterior) / totalHastaMesAnterior) * 100 : 0
      }
    });
  } catch (error) {
    console.error('Error al obtener comparación anual:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para obtener datos de empresa
router.get('/empresa/:id', middleware.authenticateToken, async (req, res) => {
  try {
    const empresaId = req.params.id;
    
    console.log(`🏢 Obteniendo datos de empresa ${empresaId}...`);
    
    // Verificar que el usuario pertenece a esta empresa (opcional - comentado para testing)
    // if (req.user.id_empresa !== parseInt(empresaId)) {
    //   return res.status(403).json({
    //     success: false,
    //     message: 'No tienes permisos para ver esta empresa'
    //   });
    // }

    const query = `
      SELECT 
        e.id,
        e.nombre,
        e.rut,
        e.direccion,
        e.activa,
        e.fecha_creacion,
        e.fecha_actualizacion
      FROM empresas e
      WHERE e.id = ?
    `;
    
    const empresas = await helpers.query(query, [empresaId]);
    
    if (empresas.length === 0) {
      console.log(`❌ Empresa ${empresaId} no encontrada`);
      return res.status(404).json({
        success: false,
        message: 'Empresa no encontrada'
      });
    }
    
    console.log(`✅ Empresa encontrada: ${empresas[0].nombre}`);
    
    res.json({ 
      success: true,
      data: empresas[0],
      message: 'Datos de empresa obtenidos exitosamente'
    });
  } catch (error) {
    console.error('❌ Error al obtener datos de empresa:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoints para estadísticas del centro de operaciones
router.get('/estadisticas/motos-stock', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_empresa } = req.query;
    
    if (!id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro id_empresa es requerido'
      });
    }

    const query = `
      SELECT COUNT(*) as cantidad
      FROM motos m
      JOIN estado_comercial ec ON m.id_moto = ec.id_moto
      WHERE m.id_empresa = ? 
      AND ec.estado IN ('DISPONIBLE', 'RESERVADA')
    `;
    
    const result = await helpers.query(query, [id_empresa]);
    
    res.json({ 
      success: true,
      cantidad: result[0]?.cantidad || 0
    });
  } catch (error) {
    console.error('Error obteniendo motos en stock:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

router.get('/estadisticas/motos-consignadas', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_empresa } = req.query;
    
    if (!id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro id_empresa es requerido'
      });
    }

    const query = `
      SELECT COUNT(*) as cantidad
      FROM motos m
      JOIN estado_comercial ec ON m.id_moto = ec.id_moto
      WHERE m.id_empresa = ? 
      AND ec.origen = 'CONSIGNACION'
      AND ec.estado = 'DISPONIBLE'
    `;
    
    const result = await helpers.query(query, [id_empresa]);
    
    res.json({ 
      success: true,
      cantidad: result[0]?.cantidad || 0
    });
  } catch (error) {
    console.error('Error obteniendo motos consignadas:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

router.get('/estadisticas/ventas-hoy', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_empresa } = req.query;
    
    if (!id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro id_empresa es requerido'
      });
    }

    const query = `
      SELECT COUNT(*) as cantidad
      FROM ventas v
      WHERE v.id_empresa = ? 
      AND DATE(v.fecha_venta) = CURDATE()
    `;
    
    const result = await helpers.query(query, [id_empresa]);
    
    res.json({ 
      success: true,
      cantidad: result[0]?.cantidad || 0
    });
  } catch (error) {
    console.error('Error obteniendo ventas de hoy:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

router.get('/estadisticas/clientes-total', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_empresa } = req.query;
    
    if (!id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro id_empresa es requerido'
      });
    }

    const query = `
      SELECT COUNT(*) as cantidad
      FROM clientes c
      WHERE c.id_empresa = ?
    `;
    
    const result = await helpers.query(query, [id_empresa]);
    
    res.json({ 
      success: true,
      cantidad: result[0]?.cantidad || 0
    });
  } catch (error) {
    console.error('Error obteniendo total de clientes:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para obtener motos disponibles para venta
router.get('/motos/disponibles', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_empresa } = req.query;
    
    if (!id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro id_empresa es requerido'
      });
    }

    const query = `
      SELECT 
        m.id_moto,
        m.año,
        m.color,
        m.kms,
        m.patente,
        m.soc,
        m.llaves,
        m.observaciones,
        
        ec.precio_publicacion,
        ec.estado,
        ec.condicion,
        ec.origen,
        ec.fecha_ingreso,
        
        mo.nombre as modelo,
        mo.categoria,
        
        ma.nombre as marca,
        ma.país_origen
        
      FROM motos m
      JOIN estado_comercial ec ON m.id_moto = ec.id_moto
      LEFT JOIN modelos mo ON m.id_modelo = mo.id
      LEFT JOIN marcas ma ON mo.id_marca = ma.id
      WHERE m.id_empresa = ?
      AND ec.estado IN ('DISPONIBLE', 'RESERVADA')
      ORDER BY ec.fecha_ingreso DESC
    `;
    
    const motos = await helpers.query(query, [id_empresa]);
    
    // Transformar los datos para el frontend
    const motosFormateadas = motos.map(moto => ({
      id_moto: moto.id_moto,
      marca: moto.marca || 'Sin marca',
      modelo: moto.modelo || 'Sin modelo',
      año: moto.año,
      anio: moto.año, // Alias para compatibilidad
      color: moto.color,
      kms: moto.kms,
      kilometraje: moto.kms, // Alias para compatibilidad
      patente: moto.patente,
      soc: moto.soc,
      llaves: moto.llaves,
      precio: moto.precio_publicacion,
      precio_venta: moto.precio_publicacion, // Alias para compatibilidad
      estado: moto.estado,
      condicion: moto.condicion,
      origen: moto.origen,
      observaciones: moto.observaciones,
      fecha_ingreso: moto.fecha_ingreso,
      categoria: moto.categoria,
      pais_origen: moto.país_origen
    }));
    
    res.json(motosFormateadas);
  } catch (error) {
    console.error('Error obteniendo motos disponibles:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para buscar cliente por cédula/RUT
router.get('/clientes/cedula/:cedula', middleware.authenticateToken, async (req, res) => {
  try {
    const { cedula } = req.params;
    const { id_empresa } = req.query;
    
    if (!id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro id_empresa es requerido'
      });
    }

    const query = `
      SELECT 
        c.id,
        c.nombres,
        c.apellidos,
        c.cedula,
        c.tipo_cliente,
        c.id_region,
        c.edad,
        c.telefono,
        c.email,
        c.direccion,
        c.fecha_creacion,
        r.nombre as region_nombre
      FROM clientes c
      LEFT JOIN regiones r ON c.id_region = r.id
      WHERE c.cedula = ? AND c.id_empresa = ?
      LIMIT 1
    `;
    
    const result = await helpers.query(query, [cedula, id_empresa]);
    
    if (result.length > 0) {
      res.json(result[0]);
    } else {
      res.status(404).json({
        success: false,
        message: 'Cliente no encontrado'
      });
    }
  } catch (error) {
    console.error('Error buscando cliente por cédula:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para buscar clientes por término (nombre, apellido, RUT, teléfono)
router.get('/clientes/buscar', middleware.authenticateToken, async (req, res) => {
  try {
    const { termino, id_empresa } = req.query;
    
    if (!id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro id_empresa es requerido'
      });
    }
    
    if (!termino || termino.length < 2) {
      return res.json([]);
    }
    
    const query = `
      SELECT 
        c.id,
        c.nombres,
        c.apellidos,
        c.cedula,
        c.tipo_cliente,
        c.id_region,
        c.edad,
        c.telefono,
        c.email,
        c.direccion,
        c.fecha_creacion,
        r.nombre as region_nombre
      FROM clientes c
      LEFT JOIN regiones r ON c.id_region = r.id
      WHERE c.id_empresa = ? 
        AND (
          c.nombres LIKE ? 
          OR c.apellidos LIKE ? 
          OR c.cedula LIKE ? 
          OR c.telefono LIKE ?
          OR CONCAT(c.nombres, ' ', c.apellidos) LIKE ?
        )
      ORDER BY c.nombres ASC, c.apellidos ASC
      LIMIT 10
    `;
    
    const searchTerm = `%${termino}%`;
    const result = await helpers.query(query, [
      id_empresa, 
      searchTerm, 
      searchTerm, 
      searchTerm, 
      searchTerm, 
      searchTerm
    ]);
    
    res.json(result);
  } catch (error) {
    console.error('Error buscando clientes:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para crear nuevo cliente
router.post('/clientes', middleware.authenticateToken, async (req, res) => {
  try {
    const { 
      cedula, 
      nombres, 
      apellidos, 
      tipo_cliente, 
      id_region, 
      edad, 
      telefono, 
      email, 
      direccion, 
      id_empresa 
    } = req.body;
    
    if (!cedula || !nombres || !apellidos || !tipo_cliente || !id_region || !id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'Los campos cedula, nombres, apellidos, tipo_cliente, id_region e id_empresa son obligatorios'
      });
    }
    
    // Verificar si el cliente ya existe
    const existeQuery = `
      SELECT id FROM clientes WHERE cedula = ? AND id_empresa = ?
    `;
    const existeResult = await helpers.query(existeQuery, [cedula, id_empresa]);
    
    if (existeResult.length > 0) {
      return res.status(409).json({
        success: false,
        message: 'Ya existe un cliente registrado con esta cédula'
      });
    }
    
    // Insertar nuevo cliente
    const insertQuery = `
      INSERT INTO clientes (
        id_empresa,
        cedula,
        nombres,
        apellidos,
        tipo_cliente,
        id_region,
        edad,
        telefono,
        email,
        direccion,
        fecha_creacion
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
    `;
    
    const result = await helpers.query(insertQuery, [
      id_empresa,
      cedula,
      nombres,
      apellidos,
      tipo_cliente,
      id_region,
      edad || null,
      telefono || null,
      email || null,
      direccion || null
    ]);
    
    if (result.insertId) {
      // Obtener el cliente recién creado con información completa
      const clienteQuery = `
        SELECT 
          c.id,
          c.nombres,
          c.apellidos,
          c.cedula,
          c.tipo_cliente,
          c.id_region,
          c.edad,
          c.telefono,
          c.email,
          c.direccion,
          c.fecha_creacion,
          r.nombre as region_nombre
        FROM clientes c
        LEFT JOIN regiones r ON c.id_region = r.id
        WHERE c.id = ?
      `;
      
      const clienteCreado = await helpers.query(clienteQuery, [result.insertId]);
      
      res.status(201).json({
        success: true,
        message: 'Cliente creado exitosamente',
        cliente: clienteCreado[0],
        id: result.insertId
      });
    } else {
      throw new Error('No se pudo crear el cliente');
    }
    
  } catch (error) {
    console.error('Error creando cliente:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para crear/registrar logs de operaciones
router.post('/logs', middleware.authenticateToken, async (req, res) => {
  try {
    const { tipo_operacion, tabla_afectada, id_registro, detalle } = req.body;
    const id_usuario = req.user?.userId || req.user?.id;
    
    if (!tipo_operacion || !tabla_afectada || !detalle) {
      return res.status(400).json({
        success: false,
        message: 'Faltan campos obligatorios para el log'
      });
    }

    const query = `
      INSERT INTO log_operaciones 
      (id_usuario, tipo_operacion, tabla_afectada, id_registro, detalle, fecha_operacion)
      VALUES (?, ?, ?, ?, ?, NOW())
    `;
    
    const result = await helpers.query(query, [
      id_usuario,
      tipo_operacion,
      tabla_afectada,
      id_registro || null,
      detalle
    ]);
    
    res.json({ 
      success: true,
      message: 'Log registrado correctamente',
      id_log: result.insertId
    });
  } catch (error) {
    console.error('Error registrando log:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Health check endpoint
router.get('/health', async (req, res) => {
  try {
    await helpers.query('SELECT 1');
    const status = {
      success: true,
      message: 'Servidor funcionando correctamente',
      timestamp: new Date().toISOString(),
      database: 'conectada',
      version: '1.0.0',
      uploads: {
        directory: path.join(__dirname, 'uploads'),
        exists: fs.existsSync(path.join(__dirname, 'uploads'))
      }
    };
    res.json(status);
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Error de conexión a la base de datos',
      error: error.message,
      timestamp: new Date().toISOString()
    });
  }
});

// Endpoint simplificado para motos (usando solo tablas motos y estado_comercial)
router.get('/motos/simple', middleware.authenticateToken, async (req, res) => {
  try {
    console.log('🏍️ Fetching motos data from tables: motos + estado_comercial');
    
    const limite = parseInt(req.query.limite) || 10;
    const offset = parseInt(req.query.offset) || 0;
    const empresa = req.query.empresa || req.user.id_empresa;
    
    console.log(`📊 Parámetros: empresa=${empresa}, limite=${limite}, offset=${offset}`);
    
    // Query usando SOLO las tablas motos y estado_comercial con campos exactos
    const query = `
      SELECT 
        -- Campos de tabla motos (exactos según BD)
        m.id_moto as id,
        m.id_empresa,
        m.id_modelo,
        m.año,
        m.color,
        m.kms,
        m.patente,
        m.soc,
        m.llaves,
        m.observaciones,
        m.fecha_creacion,
        m.fecha_actualizacion,
        
        -- Campos de tabla estado_comercial (exactos según BD)
        ec.id as estado_id,
        ec.precio_publicacion as precio,
        ec.estado,
        ec.condicion,
        ec.origen,
        ec.id_toma,
        ec.fecha_toma,
        ec.fecha_ingreso,
        ec.fecha_salida,
        ec.creado_por,
        ec.modificado_por,
        
        -- Campos temporales hasta llenar marcas/modelos
        CONCAT('Modelo ', m.id_modelo) as modelo,
        'Marca pendiente' as marca
        
      FROM motos m
      LEFT JOIN estado_comercial ec ON m.id_moto = ec.id_moto
      WHERE m.id_empresa = ?
      ORDER BY m.fecha_creacion DESC
      LIMIT ? OFFSET ?
    `;
    
    const result = await helpers.query(query, [empresa, limite, offset]);
    
    console.log(`✅ Query ejecutado: ${result.length} motos encontradas para empresa ${empresa}`);
    
    res.json({
      success: true,
      data: result,
      total: result.length,
      empresa: empresa,
      message: `${result.length} motos obtenidas exitosamente para empresa ${empresa}`
    });
    
  } catch (error) {
    console.error('❌ Error en endpoint motos/simple:', error);
    res.status(500).json({
      success: false,
      message: 'Error al obtener motos',
      error: error.message
    });
  }
});

// Endpoint principal de motos con relaciones completas
router.get('/motos', middleware.authenticateToken, async (req, res) => {
  try {
    console.log('🏍️ Fetching motos data with full relationships');
    
    const limite = parseInt(req.query.limite) || 50;
    const offset = parseInt(req.query.offset) || 0;
    const empresa = req.query.empresa || req.user.id_empresa;
    const includes = req.query.includes || '';
    
    console.log(`📊 Parámetros: empresa=${empresa}, limite=${limite}, offset=${offset}, includes=${includes}`);
    
    // Query con todas las relaciones usando JOINs incluyendo usuario_toma, precio de venta y conteo de dueños
    const query = `
      SELECT 
        -- Campos de tabla motos
        m.id_moto,
        m.id_empresa,
        m.id_modelo,
        m.año,
        m.color,
        m.kms,
        m.patente,
        m.soc,
        m.llaves,
        m.observaciones,
        
        -- Campos de estado_comercial
        ec.id as estado_comercial_id,
        ec.precio_publicacion,
        ec.estado,
        ec.condicion,
        ec.origen,
        ec.id_toma,
        ec.fecha_toma,
        ec.fecha_ingreso,
        ec.fecha_salida,
        
        -- Campos de modelo
        mo.id as modelo_id,
        mo.nombre as modelo_nombre,
        mo.categoria as modelo_categoria,
        mo.activo as modelo_activo,
        
        -- Campos de marca
        ma.id as marca_id,
        ma.nombre as marca_nombre,
        ma.país_origen as marca_pais_origen,
        ma.activa as marca_activa,
        
        -- Campos de usuario que hizo la toma
        ut.id as usuario_toma_id,
        ut.nombre as usuario_toma_nombre,
        ut.correo as usuario_toma_correo,
        ut.rol as usuario_toma_rol,
        
        -- Campos de venta (si existe)
        v.id as venta_id,
        v.precio_venta,
        v.fecha_venta,
        v.id_vendedor,
        uv.nombre as vendedor_nombre,
        
        -- Contar dueños previos
        COALESCE(hdm.total_duenos, 0) as total_duenos
        
      FROM motos m
      LEFT JOIN estado_comercial ec ON m.id_moto = ec.id_moto
      LEFT JOIN modelos mo ON m.id_modelo = mo.id
      LEFT JOIN marcas ma ON mo.id_marca = ma.id
      LEFT JOIN usuarios ut ON ec.id_toma = ut.id
      LEFT JOIN ventas v ON m.id_moto = v.id_moto
      LEFT JOIN usuarios uv ON v.id_vendedor = uv.id
      LEFT JOIN (
        SELECT id_moto, COUNT(*) as total_duenos 
        FROM historial_dueños_motos 
        GROUP BY id_moto
      ) hdm ON m.id_moto = hdm.id_moto
      WHERE m.id_empresa = ?
      ORDER BY ec.fecha_ingreso DESC, m.id_moto DESC
      LIMIT ? OFFSET ?
    `;
    
    const rows = await helpers.query(query, [empresa, limite, offset]);
    
    console.log(`✅ Query ejecutado: ${rows.length} motos encontradas para empresa ${empresa}`);
    
    // Transformar los datos para que coincidan con el formato esperado por el frontend
    const motosTransformadas = rows.map(row => ({
      id_moto: row.id_moto,
      id_empresa: row.id_empresa,
      id_modelo: row.id_modelo,
      año: row.año,
      color: row.color,
      kms: row.kms,
      patente: row.patente,
      soc: row.soc,
      llaves: row.llaves,
      observaciones: row.observaciones,
      fecha_ingreso: row.fecha_ingreso,
      total_duenos: row.total_duenos,
      
      // Objeto modelo con su marca anidada
      modelo: row.modelo_id ? {
        id: row.modelo_id,
        nombre: row.modelo_nombre,
        categoria: row.modelo_categoria,
        activo: row.modelo_activo,
        marca: row.marca_id ? {
          id: row.marca_id,
          nombre: row.marca_nombre,
          país_origen: row.marca_pais_origen,
          activa: row.marca_activa
        } : null
      } : null,
      
      // Objeto estado_comercial con usuario_toma incluido
      estado_comercial: row.estado_comercial_id ? {
        id: row.estado_comercial_id,
        precio_publicacion: row.precio_publicacion,
        precio_toma: row.precio_venta, // Usar precio de venta como precio_toma
        estado: row.estado,
        condicion: row.condicion,
        origen: row.origen,
        id_toma: row.id_toma,
        fecha_toma: row.fecha_toma,
        fecha_ingreso: row.fecha_ingreso,
        fecha_salida: row.fecha_salida,
        usuario_toma: row.usuario_toma_id ? {
          id: row.usuario_toma_id,
          nombre: row.usuario_toma_nombre,
          correo: row.usuario_toma_correo,
          rol: row.usuario_toma_rol
        } : null
      } : null,
      
      // Información de venta (si existe)
      venta: row.venta_id ? {
        id: row.venta_id,
        precio_venta: row.precio_venta,
        fecha_venta: row.fecha_venta,
        vendedor: row.vendedor_nombre
      } : null
    }));
    
    console.log(`🔍 Primera moto transformada:`, JSON.stringify(motosTransformadas[0], null, 2));
    
    res.json({ 
      success: true,
      data: motosTransformadas,
      total: motosTransformadas.length,
      empresa: empresa,
      message: `${motosTransformadas.length} motos obtenidas exitosamente para empresa ${empresa}`
    });
  } catch (error) {
    console.error('❌ Error al obtener motos:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// CORS preflight para login
router.options('/login', (req, res) => {
  res.header('Access-Control-Allow-Origin', '*');
  res.header('Access-Control-Allow-Methods', 'POST, OPTIONS');
  res.header('Access-Control-Allow-Headers', 'Content-Type, Authorization');
  res.sendStatus(200);
});

// Register/Signup endpoint
router.post('/signup', 
  middleware.validateInput({
    nombre: { required: true, minLength: 2, maxLength: 100 },
    correo: { required: true, type: 'email' },
    clave: { required: true, type: 'password' },
    id_empresa: { required: true, type: 'number' }
  }),
  async (req, res) => {
    const { nombre, correo, clave, id_empresa } = req.body;
    
    try {
      // Verificar si el usuario ya existe
      const rows = await helpers.query(
        'SELECT id FROM usuarios WHERE correo = ?',
        [correo]
      );
      
      if (rows.length > 0) {
        return res.status(400).json({ 
          success: false, 
          message: 'El usuario ya existe' 
        });
      }
      
      // Verificar que la empresa existe
      const empresa = await helpers.query(
        'SELECT id FROM empresas WHERE id = ?',
        [id_empresa]
      );
      
      if (empresa.length === 0) {
        return res.status(400).json({
          success: false,
          message: 'Empresa no válida'
        });
      }
      
      // Hash de la contraseña
      const hashedPassword = await bcrypt.hash(clave, 10);
      
      // Verificar que la empresa esté activa
      const empresaActiva = await helpers.query(
        'SELECT activa FROM empresas WHERE id = ? AND activa = TRUE',
        [id_empresa]
      );

      if (empresaActiva.length === 0) {
        return res.status(400).json({
          success: false,
          message: 'La empresa no está activa'
        });
      }

      // Insertar nuevo usuario con rol VENDEDOR por defecto
      const result = await helpers.query(
        'INSERT INTO usuarios (nombre, correo, clave, id_empresa, rol, activo) VALUES (?, ?, ?, ?, ?, TRUE)',
        [nombre, correo, hashedPassword, id_empresa, 'VENDEDOR']
      );
      
      res.status(201).json({
        success: true,
        message: 'Usuario creado exitosamente',
        userId: result.insertId
      });
      
    } catch (error) {
      console.error('Error en signup:', error);
      res.status(500).json({ 
        success: false, 
        message: 'Error interno del servidor' 
      });
    }
  }
);

// Login endpoint
router.post('/login', 
  middleware.validateInput({
    correo: { required: true, type: 'email' },
    clave: { required: true, minLength: 1 }
  }),
  async (req, res) => {
    const { correo, clave } = req.body;
    console.log('👤 Intento de login para:', correo);
    
    try {
      // Buscar usuario con información de empresa
      console.log('🔍 Buscando usuario en la base de datos...');
      const users = await helpers.query(`
        SELECT 
          u.id,
          u.nombre,
          u.correo,
          u.clave,
          u.rol,
          u.id_empresa,
          u.activo,
          e.nombre as empresa_nombre,
          e.activa as empresa_activa
        FROM usuarios u
        LEFT JOIN empresas e ON u.id_empresa = e.id
        WHERE u.correo = ?
      `, [correo]);
      
      console.log('📊 Resultados encontrados:', users.length);
      
      if (users.length === 0) {
        console.log('❌ Usuario no encontrado');
        return res.status(401).json({
          success: false,
          message: 'Credenciales inválidas'
        });
      }
      
      // Si encontramos el usuario, verificamos su estado
      const user = users[0];
      console.log('✅ Usuario encontrado, verificando estado...');

      // Verificar que el usuario esté activo
      if (!user.activo) {
        console.log('❌ Usuario inactivo');
        return res.status(401).json({
          success: false,
          message: 'Usuario inactivo'
        });
      }

      // Verificar que la empresa esté activa
      if (!user.empresa_activa) {
        console.log('❌ Empresa inactiva');
        return res.status(401).json({
          success: false,
          message: 'Empresa inactiva'
        });
      }

      console.log('✅ Usuario y empresa activos, verificando contraseña...');
      
      // Debug: mostrar hash almacenado (solo en desarrollo)
      if (process.env.NODE_ENV === 'development') {
        console.log('🔐 Hash almacenado:', user.clave);
      }
      
      // Verificar contraseña
      let passwordValid = false;
      
      // Si es el admin por defecto
      if (correo === 'juan.admin@motoexperts.cl' && clave === 'MotoAdmin2025#') {
        console.log('🔑 Validando admin por defecto');
        passwordValid = true;
        // Asignar rol de ADMIN si no está establecido
        if (user.rol !== 'ADMIN') {
          await helpers.query('UPDATE usuarios SET rol = ? WHERE id = ?', ['ADMIN', user.id]);
          user.rol = 'ADMIN';
        }
      } else {
        try {
          passwordValid = await bcrypt.compare(clave, user.clave);
          console.log('🔒 Resultado de verificación bcrypt:', passwordValid);
        } catch (error) {
          console.error('❌ Error al verificar contraseña:', error);
          passwordValid = false;
        }
      }
      
      if (!passwordValid) {
        console.log('❌ Contraseña inválida');
        return res.status(401).json({
          success: false,
          message: 'Credenciales inválidas'
        });
      }
      
      console.log('✅ Contraseña verificada, generando token...');
      
      // Ya tenemos la información de la empresa del JOIN inicial
      // Preparar datos del usuario (sin la contraseña)
      const userData = {
        id: user.id,
        nombre: user.nombre,
        correo: user.correo,
        rol: user.rol,
        id_empresa: user.id_empresa,
        empresa_nombre: user.empresa_nombre
      };
      
      // Generar JWT token con información adicional de seguridad
      const token = jwt.sign(
        { 
          userId: userData.id, 
          correo: userData.correo, 
          rol: userData.rol,
          id_empresa: userData.id_empresa,
          activo: user.activo,
          empresa_activa: user.empresa_activa,
          timestamp: new Date().getTime()
        },
        JWT_SECRET,
        { 
          expiresIn: '24h',
          algorithm: 'HS256'
        }
      );
      
      // Remover la contraseña de la respuesta
      delete userData.clave;
      
      console.log('✅ Login exitoso para:', userData.correo);
      
      res.json({
        success: true,
        message: 'Login exitoso',
        token,
        user: userData
      });
      
    } catch (error) {
      console.error('❌ Error en login:', error);
      console.error('📜 Stack trace:', error.stack);
      
      // Mensaje de error más específico
      let errorMessage = 'Error interno del servidor';
      let errorDetails = undefined;
      
      if (error.code === 'ER_NO_SUCH_TABLE') {
        errorMessage = 'Error de configuración: Tabla no encontrada';
        errorDetails = `Tabla requerida: ${error.message.split("'")[1]}`;
      } else if (error.code === 'ER_BAD_FIELD_ERROR') {
        errorMessage = 'Error de configuración: Campo no encontrado';
        errorDetails = error.message;
      } else if (error.code === 'ECONNREFUSED') {
        errorMessage = 'No se pudo conectar a la base de datos';
        errorDetails = 'Verifica que MySQL esté ejecutándose';
      }
      
      console.error('📝 Detalles del error:', { 
        message: errorMessage, 
        code: error.code,
        details: errorDetails 
      });
      
      res.status(500).json({ 
        success: false, 
        message: errorMessage,
        details: process.env.NODE_ENV === 'development' ? errorDetails : undefined,
        error: process.env.NODE_ENV === 'development' ? error.message : undefined
      });
    }
  }
);

// Endpoint para obtener clientes con regiones
router.get('/clientes', async (req, res) => {
  const idEmpresa = req.query.empresa;
  if (!idEmpresa) {
    return res.status(400).json({ error: 'ID de empresa es requerido' });
  }

  try {
    const clientes = await db.query(
      `SELECT c.*, r.nombre AS region_nombre 
       FROM clientes c 
       LEFT JOIN regiones r ON c.id_region = r.id 
       WHERE c.id_empresa = ?`,
      [idEmpresa]
    );
    res.json({ data: clientes });
  } catch (error) {
    console.error('Error al obtener clientes:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para obtener regiones
router.get('/regiones', async (req, res) => {
  try {
    const regiones = await helpers.query('SELECT * FROM regiones WHERE activa = TRUE');
    res.json(regiones);
  } catch (error) {
    console.error('Error al obtener regiones:', error);
    res.status(500).json({ error: 'Error interno del servidor' });
  }
});

// Endpoint para obtener marcas
router.get('/marcas', async (req, res) => {
  try {
    const marcas = await helpers.query('SELECT * FROM marcas ORDER BY nombre ASC');
    res.json(marcas);
  } catch (error) {
    console.error('Error al obtener marcas:', error);
    res.status(500).json({ error: 'Error interno del servidor' });
  }
});

// Endpoint para obtener modelos por marca
router.get('/modelos', async (req, res) => {
  try {
    const { marca } = req.query;
    
    if (!marca) {
      return res.status(400).json({ 
        success: false, 
        error: 'El parámetro marca es requerido' 
      });
    }
    
    const modelos = await helpers.query(
      'SELECT * FROM modelos WHERE id_marca = ? ORDER BY nombre ASC',
      [marca]
    );
    
    res.json(modelos);
  } catch (error) {
    console.error('Error al obtener modelos:', error);
    res.status(500).json({ error: 'Error interno del servidor' });
  }
});

// Endpoint para obtener regiones
router.get('/regiones', async (req, res) => {
  try {
    const regiones = await helpers.query('SELECT * FROM regiones WHERE activa = 1 ORDER BY nombre ASC');
    res.json(regiones);
  } catch (error) {
    console.error('Error al obtener regiones:', error);
    res.status(500).json({ error: 'Error interno del servidor' });
  }
});

// Endpoint para crear ventas
router.post('/ventas', middleware.authenticateToken, async (req, res) => {
  const connection = await pool.promise().getConnection();
  
  try {
    await connection.beginTransaction();
    
    const {
      id_empresa,
      id_moto,
      id_cliente,
      id_vendedor,
      precio_venta,
      descuento,
      metodo_pago,
      observaciones,
      fecha_venta
    } = req.body;

    // Validaciones básicas
    if (!id_empresa || !id_moto || !id_cliente || !precio_venta) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'Faltan campos obligatorios para crear la venta'
      });
    }

    // El id_vendedor viene del frontend o del token JWT
    const vendedor_id = id_vendedor || req.user?.id || 1;

    // VERIFICAR ORIGEN DE LA MOTO PARA CALCULAR COMISIÓN
    // Solo las motos en consignación generan comisión
    const origenQuery = `
      SELECT origen FROM estado_comercial WHERE id_moto = ?
    `;
    const [origenResult] = await connection.execute(origenQuery, [id_moto]);
    
    let comision = 0;
    if (origenResult.length > 0 && origenResult[0].origen === 'CONSIGNACION') {
      // Solo calcular comisión del 5% si es consignación
      comision = precio_venta * 0.05;
      console.log(`💰 Moto en CONSIGNACION - Calculando comisión: $${comision.toLocaleString('es-CL')}`);
    } else {
      console.log(`🏢 Moto PROPIA/PARTE_DE_PAGO - Sin comisión`);
    }

    const fecha_venta_final = fecha_venta || new Date().toISOString().split('T')[0];

    // 1. Insertar la venta
    const ventaQuery = `
      INSERT INTO ventas (
        id_empresa, id_moto, id_cliente, id_vendedor, precio_venta, 
        descuento, comision, metodo_pago, observaciones, fecha_venta, creado_por
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;

    const ventaResult = await connection.execute(ventaQuery, [
      id_empresa,
      id_moto,
      id_cliente,
      vendedor_id,
      precio_venta,
      descuento || 0,
      comision,
      metodo_pago || 'EFECTIVO',
      observaciones || null,
      fecha_venta_final,
      vendedor_id
    ]);

    const id_venta = ventaResult[0].insertId;

    // 2. Actualizar estado de la moto a VENDIDA
    const updateEstadoQuery = `
      UPDATE estado_comercial 
      SET estado = 'VENDIDA', 
          fecha_salida = ?,
          modificado_por = ?
      WHERE id_moto = ?
    `;

    await connection.execute(updateEstadoQuery, [
      fecha_venta_final,
      vendedor_id,
      id_moto
    ]);

    // 3. Obtener datos del cliente para el historial
    const clienteQuery = `
      SELECT nombres, apellidos, cedula 
      FROM clientes 
      WHERE id = ?
    `;
    
    const [clienteRows] = await connection.execute(clienteQuery, [id_cliente]);
    const cliente = clienteRows[0];

    // 4. Cerrar historial anterior (si existe)
    await connection.execute(
      'UPDATE historial_dueños_motos SET fecha_fin = ?, activo = 0 WHERE id_moto = ? AND activo = 1',
      [fecha_venta_final, id_moto]
    );

    // 5. Insertar nuevo historial de dueño
    const historialQuery = `
      INSERT INTO historial_dueños_motos 
      (id_moto, rut_dueño, nombre_dueño, fecha_inicio, activo) 
      VALUES (?, ?, ?, ?, 1)
    `;

    await connection.execute(historialQuery, [
      id_moto,
      cliente.cedula,
      `${cliente.nombres} ${cliente.apellidos}`,
      fecha_venta_final
    ]);

    // 6. Obtener datos de la moto para el log
    const motoQuery = `
      SELECT m.*, mo.nombre as modelo, ma.nombre as marca
      FROM motos m
      JOIN modelos mo ON m.id_modelo = mo.id
      JOIN marcas ma ON mo.id_marca = ma.id
      WHERE m.id_moto = ?
    `;
    
    const [motoRows] = await connection.execute(motoQuery, [id_moto]);
    const moto = motoRows[0];

    // 7. Registrar en log de operaciones
    const logQuery = `
      INSERT INTO log_operaciones 
      (id_usuario, tipo_operacion, tabla_afectada, id_registro, detalle) 
      VALUES (?, 'venta', 'ventas', ?, ?)
    `;

    const detalle = `VENTA: Venta de ${moto.marca} ${moto.modelo} completada exitosamente - Precio final: $${Number(precio_venta).toLocaleString('es-CL')} | Cliente: ${cliente.nombres}`;

    await connection.execute(logQuery, [
      vendedor_id,
      id_venta,
      detalle
    ]);

    await connection.commit();

    res.status(201).json({
      success: true,
      message: 'Venta procesada exitosamente',
      data: {
        id_venta: id_venta,
        cliente: `${cliente.nombres} ${cliente.apellidos}`,
        moto: `${moto.marca} ${moto.modelo}`,
        precio_final: precio_venta - (descuento || 0)
      }
    });

  } catch (error) {
    await connection.rollback();
    console.error('Error procesando venta:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor al procesar la venta',
      error: error.message
    });
  } finally {
    connection.release();
  }
});

// Endpoint para obtener especificaciones de una moto
router.get('/especificaciones/:id_moto', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_moto } = req.params;
    
    const especificaciones = await helpers.query(
      'SELECT * FROM especificaciones_tecnicas WHERE id_moto = ?',
      [id_moto]
    );

    res.json(especificaciones[0] || null);

  } catch (error) {
    console.error('Error obteniendo especificaciones:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para obtener historial de dueños de una moto
router.get('/historial-duenos/moto/:id_moto', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_moto } = req.params;
    
    const historial = await helpers.query(
      `SELECT * FROM historial_dueños_motos 
       WHERE id_moto = ? 
       ORDER BY fecha_inicio DESC`,
      [id_moto]
    );

    res.json(historial);

  } catch (error) {
    console.error('Error obteniendo historial de dueños:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para crear historial de dueño
router.post('/historial-duenos', middleware.authenticateToken, async (req, res) => {
  try {
    const {
      id_moto,
      rut_dueño,
      nombre_dueño,
      fecha_inicio,
      activo
    } = req.body;

    if (!id_moto || !rut_dueño || !nombre_dueño) {
      return res.status(400).json({
        success: false,
        message: 'Faltan campos obligatorios'
      });
    }

    const query = `
      INSERT INTO historial_dueños_motos (id_moto, rut_dueño, nombre_dueño, fecha_inicio, activo)
      VALUES (?, ?, ?, ?, ?)
    `;

    const result = await helpers.query(query, [
      id_moto,
      rut_dueño,
      nombre_dueño,
      fecha_inicio || new Date().toISOString().split('T')[0],
      activo !== undefined ? activo : true
    ]);

    res.status(201).json({
      success: true,
      message: 'Historial de dueño creado exitosamente',
      id: result.insertId
    });

  } catch (error) {
    console.error('Error creando historial de dueño:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para desactivar historial de dueño anterior
router.put('/historial-duenos/moto/:id_moto/desactivar', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_moto } = req.params;
    
    await helpers.query(
      'UPDATE historial_dueños_motos SET activo = FALSE, fecha_fin = ? WHERE id_moto = ? AND activo = TRUE',
      [new Date().toISOString().split('T')[0], id_moto]
    );

    res.json({
      success: true,
      message: 'Historial anterior desactivado'
    });

  } catch (error) {
    console.error('Error desactivando historial:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para crear nueva moto
router.post('/motos', middleware.authenticateToken, async (req, res) => {
  try {
    const {
      id_empresa,
      id_modelo,
      año,
      color,
      kms,
      patente,
      soc,
      llaves,
      observaciones
    } = req.body;

    // Validaciones básicas
    if (!id_empresa || !id_modelo || !año || !patente) {
      return res.status(400).json({
        success: false,
        message: 'Faltan campos obligatorios: id_empresa, id_modelo, año, patente'
      });
    }

    // Verificar que la patente no exista
    const verificarPatente = `
      SELECT id_moto FROM motos WHERE patente = ? AND id_empresa = ?
    `;
    const existePatente = await helpers.query(verificarPatente, [patente, id_empresa]);
    
    if (existePatente.length > 0) {
      return res.status(400).json({
        success: false,
        message: 'Ya existe una moto con esta patente en la empresa'
      });
    }

    const query = `
      INSERT INTO motos (id_empresa, id_modelo, año, color, kms, patente, soc, llaves, observaciones, fecha_creacion)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
    `;

    const result = await helpers.query(query, [
      id_empresa,
      id_modelo,
      año,
      color || null,
      kms || 0,
      patente,
      soc || null,
      llaves || 2,
      observaciones || null
    ]);

    res.status(201).json({
      success: true,
      message: 'Moto creada exitosamente',
      id_moto: result.insertId
    });

  } catch (error) {
    console.error('Error creando moto:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para verificar si una patente existe
router.get('/motos/verificar-patente/:patente', middleware.authenticateToken, async (req, res) => {
  try {
    const { patente } = req.params;
    const { empresa } = req.query;

    const query = `
      SELECT COUNT(*) as count FROM motos 
      WHERE patente = ? AND id_empresa = ?
    `;
    
    const result = await helpers.query(query, [patente, empresa]);
    
    res.json({
      success: true,
      existe: result[0].count > 0
    });

  } catch (error) {
    console.error('Error verificando patente:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para crear estado comercial
router.post('/estado-comercial', middleware.authenticateToken, async (req, res) => {
  try {
    const {
      id_moto,
      precio_publicacion,
      estado,
      condicion,
      origen,
      fecha_ingreso,
      creado_por
    } = req.body;

    // Validaciones básicas
    if (!id_moto || !precio_publicacion || !estado || !condicion || !origen) {
      return res.status(400).json({
        success: false,
        message: 'Faltan campos obligatorios: id_moto, precio_publicacion, estado, condicion, origen'
      });
    }

    // Verificar que el origen sea válido (incluyendo PROPIA)
    const origenesValidos = ['CONSIGNACION', 'PARTE_PAGO', 'PROPIA', 'COMPRA', 'IMPORTACION'];
    if (!origenesValidos.includes(origen)) {
      return res.status(400).json({
        success: false,
        message: 'Origen no válido. Debe ser uno de: ' + origenesValidos.join(', ')
      });
    }

    const query = `
      INSERT INTO estado_comercial (
        id_moto, precio_publicacion, estado, condicion, origen, fecha_ingreso, creado_por
      )
      VALUES (?, ?, ?, ?, ?, ?, ?)
    `;

    const result = await helpers.query(query, [
      id_moto,
      precio_publicacion,
      estado,
      condicion,
      origen,
      fecha_ingreso || new Date().toISOString().split('T')[0],
      creado_por
    ]);

    res.status(201).json({
      success: true,
      message: 'Estado comercial creado exitosamente',
      id_estado: result.insertId
    });

  } catch (error) {
    console.error('Error creando estado comercial:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para crear especificaciones técnicas
router.post('/especificaciones', middleware.authenticateToken, async (req, res) => {
  try {
    const {
      id_moto,
      cilindrada_cc,
      numero_motor,
      numero_chasis,
      tipo_motor
    } = req.body;

    if (!id_moto) {
      return res.status(400).json({
        success: false,
        message: 'id_moto es obligatorio'
      });
    }

    if (!cilindrada_cc) {
      return res.status(400).json({
        success: false,
        message: 'cilindrada_cc es obligatorio'
      });
    }

    const query = `
      INSERT INTO especificaciones_tecnicas (
        id_moto, cilindrada_cc, numero_motor, numero_chasis, tipo_motor
      )
      VALUES (?, ?, ?, ?, ?)
    `;

    const result = await helpers.query(query, [
      id_moto,
      cilindrada_cc,
      numero_motor || null,
      numero_chasis || null,
      tipo_motor || null
    ]);

    res.status(201).json({
      success: true,
      message: 'Especificaciones técnicas guardadas exitosamente',
      id_especificacion: result.insertId
    });

  } catch (error) {
    console.error('Error creando especificaciones:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para obtener especificaciones técnicas de una moto
router.get('/especificaciones/:id_moto', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_moto } = req.params;

    const query = `
      SELECT * FROM especificaciones_tecnicas 
      WHERE id_moto = ?
    `;
    
    const result = await helpers.query(query, [id_moto]);

    if (result.length === 0) {
      return res.status(404).json({
        success: false,
        message: 'No se encontraron especificaciones para esta moto'
      });
    }

    res.json({
      success: true,
      data: result[0]
    });

  } catch (error) {
    console.error('Error obteniendo especificaciones:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para crear equipamiento adicional
router.post('/equipamiento-adicional', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_moto, tipo_equipamiento, descripcion, estado, precio } = req.body;
    
    if (!id_moto || !tipo_equipamiento || !descripcion || !estado) {
      return res.status(400).json({
        success: false,
        message: 'Faltan campos requeridos: id_moto, tipo_equipamiento, descripcion, estado'
      });
    }

    const query = `
      INSERT INTO equipamiento_adicional (
        id_moto,
        tipo_equipamiento,
        descripcion,
        estado,
        precio,
        fecha_instalacion,
        activo
      ) VALUES (?, ?, ?, ?, ?, NOW(), 1)
    `;

    const result = await helpers.query(query, [
      id_moto,
      tipo_equipamiento,
      descripcion,
      estado,
      precio || 0
    ]);

    res.status(201).json({
      success: true,
      message: 'Equipamiento adicional guardado exitosamente',
      id_equipamiento: result.insertId
    });

  } catch (error) {
    console.error('Error creando equipamiento adicional:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para actualizar estado comercial de moto
router.put('/estado-comercial/moto/:id_moto', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_moto } = req.params;
    const { estado } = req.body;
    
    const query = `
      UPDATE estado_comercial 
      SET estado = ?, fecha_salida = ?
      WHERE id_moto = ?
    `;

    await helpers.query(query, [estado, new Date().toISOString().split('T')[0], id_moto]);

    res.json({
      success: true,
      message: 'Estado comercial actualizado'
    });

  } catch (error) {
    console.error('Error actualizando estado comercial:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para obtener una moto específica por ID con todos los datos
router.get('/motos/:id', middleware.authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const empresa = req.query.empresa || req.user.id_empresa;
    
    console.log(`🏍️ Buscando moto ID ${id} para empresa ${empresa}`);
    
    // Query principal con todas las relaciones incluyendo usuario_toma y venta
    const queryPrincipal = `
      SELECT 
        -- Campos de tabla motos
        m.id_moto,
        m.id_empresa,
        m.id_modelo,
        m.año,
        m.color,
        m.kms,
        m.patente,
        m.soc,
        m.llaves,
        m.observaciones,
        m.fecha_creacion,
        m.fecha_actualizacion,
        
        -- Campos de estado_comercial
        ec.id as estado_comercial_id,
        ec.precio_publicacion,
        ec.estado,
        ec.condicion,
        ec.origen,
        ec.id_toma,
        ec.fecha_toma,
        ec.fecha_ingreso,
        ec.fecha_salida,
        ec.creado_por as ec_creado_por,
        ec.modificado_por as ec_modificado_por,
        
        -- Campos de modelo
        mo.id as modelo_id,
        mo.nombre as modelo_nombre,
        mo.categoria as modelo_categoria,
        mo.activo as modelo_activo,
        
        -- Campos de marca
        ma.id as marca_id,
        ma.nombre as marca_nombre,
        ma.país_origen as marca_pais_origen,
        ma.activa as marca_activa,
        
        -- Campos de especificaciones técnicas (solo las que existen)
        et.cilindrada_cc,
        et.numero_motor,
        et.numero_chasis,
        et.tipo_motor,
        
        -- Usuario que hizo la toma (relación con usuarios)
        ut.id as usuario_toma_id,
        ut.nombre as usuario_toma_nombre,
        ut.correo as usuario_toma_correo,
        ut.rol as usuario_toma_rol,
        
        -- Información de venta si está vendida
        v.id as venta_id,
        v.precio_venta,
        v.fecha_venta,
        v.id_vendedor,
        v.id_cliente,
        
        -- Vendedor
        uv.nombre as vendedor_nombre,
        
        -- Cliente
        c.nombres as cliente_nombres,
        c.apellidos as cliente_apellidos
        
      FROM motos m
      LEFT JOIN estado_comercial ec ON m.id_moto = ec.id_moto
      LEFT JOIN modelos mo ON m.id_modelo = mo.id
      LEFT JOIN marcas ma ON mo.id_marca = ma.id
      LEFT JOIN especificaciones_tecnicas et ON m.id_moto = et.id_moto
      LEFT JOIN usuarios ut ON ec.id_toma = ut.id
      LEFT JOIN ventas v ON m.id_moto = v.id_moto
      LEFT JOIN usuarios uv ON v.id_vendedor = uv.id
      LEFT JOIN clientes c ON v.id_cliente = c.id
      WHERE m.id_moto = ? AND m.id_empresa = ?
      LIMIT 1
    `;
    
    const result = await helpers.query(queryPrincipal, [id, empresa]);
    
    if (result.length === 0) {
      console.log(`❌ Moto ID ${id} no encontrada para empresa ${empresa}`);
      return res.status(404).json({
        success: false,
        message: 'Moto no encontrada'
      });
    }
    
    const row = result[0];
    console.log(`✅ Moto encontrada: ${row.marca_nombre || 'Sin marca'} ${row.modelo_nombre || 'Sin modelo'} - ${row.patente || 'Sin patente'}`);
    console.log(`📋 SOC: ${row.soc || 'Sin SOC'}, Usuario Toma: ${row.usuario_toma_nombre || 'Sin usuario toma'} (ID: ${row.id_toma || 'N/A'})`);
    console.log(`🏢 Estado comercial: ${row.estado || 'Sin estado'}, Condición: ${row.condicion || 'Sin condición'}`);
    
    // Obtener mantenciones con información de usuarios
    const queryMantenciones = `
      SELECT 
        m.*,
        u1.nombre as creado_por_nombre,
        u2.nombre as modificado_por_nombre
      FROM mantenciones m
      LEFT JOIN usuarios u1 ON m.creado_por = u1.id
      LEFT JOIN usuarios u2 ON m.modificado_por = u2.id
      WHERE m.id_moto = ?
      ORDER BY m.fecha DESC
    `;
    
    const mantenciones = await helpers.query(queryMantenciones, [id]);
    console.log(`✅ ${mantenciones.length} mantenciones encontradas`);
    
    // Para cada mantención, obtener sus imágenes
    for (let mantencion of mantenciones) {
      const queryImagenesMant = `
        SELECT 
          id,
          nombre_archivo,
          ruta_archivo,
          tipo_imagen,
          fecha_subida
        FROM imagenes_mantenciones 
        WHERE id_mantencion = ?
        ORDER BY fecha_subida ASC
      `;
      
      const imagenesMantencion = await helpers.query(queryImagenesMant, [mantencion.id]);
      
      // Procesar rutas de imágenes de mantención
      mantencion.imagenes = imagenesMantencion.map(img => {
        const nombreArchivo = img.ruta_archivo.split(/[\\\/]/).pop();
        return {
          ...img,
          url: `/uploads/${nombreArchivo}`,
          url_completa: `${req.protocol}://${req.get('host')}/uploads/${nombreArchivo}`
        };
      });
    }
    
    // Obtener imágenes con rutas completas
    const queryImagenes = `
      SELECT 
        id,
        nombre_archivo,
        ruta_archivo,
        es_principal
      FROM imagenes_motos 
      WHERE id_moto = ?
      ORDER BY es_principal DESC, id ASC
    `;
    
    const imagenes = await helpers.query(queryImagenes, [id]);
    console.log(`✅ ${imagenes.length} imágenes encontradas`);
    
    // Debug: mostrar las primeras imágenes para verificar rutas
    if (imagenes.length > 0) {
      console.log(`🖼️ Primera imagen - Nombre: ${imagenes[0].nombre_archivo}, Ruta: ${imagenes[0].ruta_archivo}`);
    }
    
    // Procesar rutas de imágenes para que sean accesibles via web
    const imagenesConRutaCompleta = imagenes.map(img => {
      let rutaFinal;
      
      // Si ruta_archivo contiene subcarpetas (como motos/2025/08/), usar esa ruta
      if (img.ruta_archivo && img.ruta_archivo.includes('/')) {
        rutaFinal = img.ruta_archivo.replace(/\\/g, '/');
      }
      // Si ruta_archivo es solo el nombre del archivo, usar directamente
      else if (img.ruta_archivo && !img.ruta_archivo.includes('/')) {
        rutaFinal = img.ruta_archivo;
      }
      // Fallback: usar nombre_archivo si no hay ruta_archivo
      else {
        rutaFinal = img.nombre_archivo;
      }
      
      return {
        ...img,
        url: `/uploads/${rutaFinal}`,
        url_completa: `${req.protocol}://${req.get('host')}/uploads/${rutaFinal}`
      };
    });
    
    // Debug: mostrar URL generada
    if (imagenesConRutaCompleta.length > 0) {
      console.log(`🔗 URL generada para primera imagen: ${imagenesConRutaCompleta[0].url}`);
    }
    
    // Obtener historial de dueños
    const queryHistorialDuenos = `
      SELECT 
        id,
        rut_dueño,
        nombre_dueño,
        fecha_inicio,
        fecha_fin,
        activo
      FROM historial_dueños_motos 
      WHERE id_moto = ?
      ORDER BY fecha_inicio DESC
    `;
    
    const historialDuenos = await helpers.query(queryHistorialDuenos, [id]);
    console.log(`👥 ${historialDuenos.length} dueños en historial`);
    
    // Obtener documentos asociados a la moto
    const queryDocumentos = `
      SELECT 
        d.id,
        d.tipo_documento,
        d.nombre_archivo,
        d.ruta_archivo,
        d.tamaño_archivo,
        d.mime_type,
        d.descripcion,
        d.es_publico,
        d.fecha_creacion,
        u.nombre as creado_por_nombre
      FROM documentos d
      LEFT JOIN usuarios u ON d.creado_por = u.id
      WHERE d.tipo_entidad = 'MOTO' AND d.id_entidad = ?
      ORDER BY d.fecha_creacion DESC
    `;
    
    console.log(`📄 Obteniendo documentos para moto ${id}`);
    const documentos = await helpers.query(queryDocumentos, [id]);
    console.log(`✅ ${documentos.length} documentos encontrados`);
    
    // Procesar rutas de documentos para que sean accesibles via web
    const documentosConRutaCompleta = documentos.map(doc => {
      let rutaFinal;
      
      // Si ruta_archivo contiene subcarpetas (como documentos/2025/08/), usar esa ruta
      if (doc.ruta_archivo && doc.ruta_archivo.includes('/')) {
        rutaFinal = doc.ruta_archivo.replace(/\\/g, '/');
      }
      // Si ruta_archivo es solo el nombre del archivo, usar directamente  
      else if (doc.ruta_archivo && !doc.ruta_archivo.includes('/')) {
        rutaFinal = doc.ruta_archivo;
      }
      // Fallback: usar nombre_archivo si no hay ruta_archivo
      else {
        rutaFinal = doc.nombre_archivo;
      }
      
      return {
        ...doc,
        url: `/uploads/${rutaFinal}`,
        url_completa: `${req.protocol}://${req.get('host')}/uploads/${rutaFinal}`,
        tamaño_formateado: formatearTamaño(doc.tamaño_archivo)
      };
    });
    
    // Obtener equipamiento adicional
    const queryEquipamiento = `
      SELECT 
        id,
        tipo_equipamiento,
        descripcion,
        estado,
        fecha_instalacion,
        precio,
        activo
      FROM equipamiento_adicional 
      WHERE id_moto = ? AND activo = TRUE
      ORDER BY fecha_instalacion DESC
    `;
    
    const equipamiento = await helpers.query(queryEquipamiento, [id]);
    console.log(`✅ ${equipamiento.length} equipamientos encontrados`);
    
    // Función auxiliar para formatear tamaño de archivos
    function formatearTamaño(bytes) {
      if (bytes === 0) return '0 Bytes';
      const k = 1024;
      const sizes = ['Bytes', 'KB', 'MB', 'GB'];
      const i = Math.floor(Math.log(bytes) / Math.log(k));
      return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
    
    // Calcular totales
    const totalDuenos = historialDuenos.length;
    const costoTotalMantenciones = mantenciones.reduce((total, mantencion) => {
      return total + (parseFloat(mantencion.costo) || 0);
    }, 0);
    const costoEquipamiento = equipamiento.reduce((total, e) => total + (parseFloat(e.precio) || 0), 0);
    const precioBase = parseFloat(row.precio_publicacion) || 0;
    const precioTotal = precioBase + costoTotalMantenciones + costoEquipamiento;
    
    // Construir objeto completo de la moto
    const motoCompleta = {
      // Datos básicos de la moto
      id_moto: row.id_moto,
      id_empresa: row.id_empresa,
      id_modelo: row.id_modelo,
      año: row.año,
      color: row.color,
      kms: row.kms,
      patente: row.patente,
      soc: row.soc,
      llaves: row.llaves,
      observaciones: row.observaciones,
      fecha_creacion: row.fecha_creacion,
      fecha_actualizacion: row.fecha_actualizacion,
      total_duenos: totalDuenos,
      
      // Modelo con marca anidada
      modelo: row.modelo_id ? {
        id: row.modelo_id,
        nombre: row.modelo_nombre,
        categoria: row.modelo_categoria,
        activo: row.modelo_activo,
        marca: row.marca_id ? {
          id: row.marca_id,
          nombre: row.marca_nombre,
          país_origen: row.marca_pais_origen,
          activa: row.marca_activa
        } : null
      } : null,
      
      // Estado comercial con usuario de toma
      estado_comercial: row.estado_comercial_id ? {
        id: row.estado_comercial_id,
        precio_publicacion: row.precio_publicacion,
        precio_base: precioBase,
        costo_mantenciones: costoTotalMantenciones,
        costo_equipamiento: costoEquipamiento,
        precio_total: precioTotal,
        estado: row.estado,
        condicion: row.condicion,
        origen: row.origen,
        id_toma: row.id_toma,
        fecha_toma: row.fecha_toma,
        fecha_ingreso: row.fecha_ingreso,
        fecha_salida: row.fecha_salida,
        creado_por: row.ec_creado_por,
        modificado_por: row.ec_modificado_por,
        usuario_toma: row.usuario_toma_id ? {
          id: row.usuario_toma_id,
          nombre: row.usuario_toma_nombre,
          correo: row.usuario_toma_correo,
          rol: row.usuario_toma_rol
        } : null
      } : null,
      
      // Especificaciones técnicas (solo campos disponibles)
      especificaciones_tecnicas: {
        cilindrada_cc: row.cilindrada_cc,
        numero_motor: row.numero_motor,
        numero_chasis: row.numero_chasis,
        tipo_motor: row.tipo_motor
      },
      
      // Información de venta si existe
      venta: row.venta_id ? {
        id: row.venta_id,
        precio_venta: row.precio_venta,
        fecha_venta: row.fecha_venta,
        vendedor: {
          id: row.id_vendedor,
          nombre: row.vendedor_nombre
        },
        cliente: {
          id: row.id_cliente,
          nombres: row.cliente_nombres,
          apellidos: row.cliente_apellidos
        }
      } : null,
      
      // Arrays con datos relacionados
      mantenciones: mantenciones,
      imagenes: imagenesConRutaCompleta,
      equipamiento: equipamiento,
      historial_duenos: historialDuenos,
      documentos: documentosConRutaCompleta
    };
    
    console.log(`� Moto completa construida con ${mantenciones.length} mantenciones, ${imagenesConRutaCompleta.length} imágenes, ${equipamiento.length} equipamientos y ${historialDuenos.length} dueños`);
    
    res.json({ 
      success: true,
      data: motoCompleta,
      message: `Moto ${id} obtenida exitosamente con todos sus datos`
    });
    
  } catch (error) {
    console.error('❌ Error al obtener detalle de moto:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para obtener mantenciones de una moto
router.get('/motos/:id/mantenciones', middleware.authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    
    const query = `
      SELECT 
        m.*,
        u1.nombre as creado_por_nombre,
        u2.nombre as modificado_por_nombre
      FROM mantenciones m
      LEFT JOIN usuarios u1 ON m.creado_por = u1.id
      LEFT JOIN usuarios u2 ON m.modificado_por = u2.id
      WHERE m.id_moto = ?
      ORDER BY m.fecha DESC
    `;
    
    const mantenciones = await helpers.query(query, [id]);
    
    res.json({
      success: true,
      data: mantenciones
    });
    
  } catch (error) {
    console.error('Error al obtener mantenciones:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para crear una nueva mantención
router.post('/motos/:id/mantenciones', middleware.authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const {
      fecha,
      tipo_mantencion,
      descripcion,
      kilometraje,
      costo,
      realizado_por,
      factura_numero
    } = req.body;
    
    if (!fecha || !tipo_mantencion || !descripcion || kilometraje === undefined || kilometraje === null || costo === undefined || costo === null) {
      return res.status(400).json({
        success: false,
        message: 'Todos los campos son obligatorios'
      });
    }
    
    const query = `
      INSERT INTO mantenciones (
        id_moto, fecha, tipo_mantencion, descripcion, kilometraje, 
        costo, realizado_por, factura_numero, creado_por
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;
    
    const result = await helpers.query(query, [
      id,
      fecha,
      tipo_mantencion,
      descripcion,
      kilometraje,
      costo,
      realizado_por || 'No especificado',
      factura_numero || null,
      req.user?.userId || req.user?.id
    ]);
    
    res.status(201).json({
      success: true,
      message: 'Mantención creada exitosamente',
      id_mantencion: result.insertId
    });
    
  } catch (error) {
    console.error('Error al crear mantención:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para subir imágenes de mantenciones
router.post('/mantenciones/:id/imagenes', middleware.authenticateToken, upload.array('imagenes', 10), async (req, res) => {
  try {
    const { id } = req.params;
    const { tipo_imagen = 'DESPUES' } = req.body;
    const files = req.files;

    if (!files || files.length === 0) {
      return res.status(400).json({
        success: false,
        message: 'No se subieron archivos'
      });
    }

    const imagenesSubidas = [];
    const año = new Date().getFullYear();
    const mes = String(new Date().getMonth() + 1).padStart(2, '0');

    for (const file of files) {
      // Crear estructura de carpetas para mantenciones: uploads/mantenciones/YYYY/MM/
      const carpetaDestino = path.join('uploads', 'mantenciones', año.toString(), mes);
      
      // Crear carpetas si no existen
      if (!fs.existsSync(carpetaDestino)) {
        fs.mkdirSync(carpetaDestino, { recursive: true });
      }

      // Generar nombre único para el archivo
      const timestamp = Date.now();
      const extension = path.extname(file.originalname);
      const nombreArchivo = `${timestamp}-${file.originalname}`;
      const rutaCompleta = path.join(carpetaDestino, nombreArchivo);
      const rutaRelativa = `mantenciones/${año}/${mes}/${nombreArchivo}`; // Para la BD

      // Mover archivo a la ubicación final
      fs.renameSync(file.path, rutaCompleta);

      // Guardar información en la base de datos
      const queryImagen = `
        INSERT INTO imagenes_mantenciones 
        (id_mantencion, nombre_archivo, ruta_archivo, tipo_imagen)
        VALUES (?, ?, ?, ?)
      `;
      
      const resultadoImagen = await helpers.query(queryImagen, [
        id,
        nombreArchivo,
        rutaRelativa,
        tipo_imagen
      ]);

      imagenesSubidas.push({
        id: resultadoImagen.insertId,
        nombre_archivo: nombreArchivo,
        ruta_archivo: rutaRelativa,
        tipo_imagen: tipo_imagen,
        url: `/uploads/${rutaRelativa}`
      });
    }

    res.status(201).json({
      success: true,
      message: `${imagenesSubidas.length} imagen(es) subida(s) exitosamente`,
      imagenes: imagenesSubidas
    });

  } catch (error) {
    console.error('Error al subir imágenes de mantención:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor al subir imágenes'
    });
  }
});

// Endpoint para obtener historial de dueños de una moto
router.get('/motos/:id/historial-duenos', middleware.authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    
    const query = `
      SELECT 
        id,
        rut_dueño,
        nombre_dueño,
        fecha_inicio,
        fecha_fin,
        activo
      FROM historial_dueños_motos 
      WHERE id_moto = ?
      ORDER BY fecha_inicio DESC
    `;
    
    const historial = await helpers.query(query, [id]);
    
    res.json({
      success: true,
      data: historial
    });
    
  } catch (error) {
    console.error('Error al obtener historial de dueños:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para agregar dueño al historial
router.post('/motos/:id/historial-duenos', middleware.authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const { rut_dueño, nombre_dueño, fecha_inicio } = req.body;
    
    if (!rut_dueño || !nombre_dueño) {
      return res.status(400).json({
        success: false,
        message: 'RUT y nombre del dueño son obligatorios'
      });
    }
    
    // Primero, desactivar el dueño anterior si existe
    await helpers.query(
      'UPDATE historial_dueños_motos SET activo = FALSE, fecha_fin = ? WHERE id_moto = ? AND activo = TRUE',
      [new Date().toISOString().split('T')[0], id]
    );
    
    // Insertar nuevo dueño
    const query = `
      INSERT INTO historial_dueños_motos (
        id_moto, rut_dueño, nombre_dueño, fecha_inicio, activo
      ) VALUES (?, ?, ?, ?, TRUE)
    `;
    
    const result = await helpers.query(query, [
      id,
      rut_dueño,
      nombre_dueño,
      fecha_inicio || new Date().toISOString().split('T')[0]
    ]);
    
    res.status(201).json({
      success: true,
      message: 'Dueño agregado al historial exitosamente',
      id_historial: result.insertId
    });
    
  } catch (error) {
    console.error('Error al agregar dueño al historial:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// Endpoint para actualizar datos de una moto
router.put('/motos/:id', middleware.authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const {
      año,
      color,
      kms,
      patente,
      soc,
      llaves,
      observaciones,
      precio_publicacion,
      estado,
      condicion,
      origen
    } = req.body;
    
    // Actualizar tabla motos
    const queryMoto = `
      UPDATE motos 
      SET año = ?, color = ?, kms = ?, patente = ?, soc = ?, llaves = ?, observaciones = ?
      WHERE id_moto = ?
    `;
    
    await helpers.query(queryMoto, [
      año, color, kms, patente, soc, llaves, observaciones, id
    ]);
    
    // Actualizar estado comercial si se proporcionan datos
    if (precio_publicacion || estado || condicion || origen) {
      const queryEstado = `
        UPDATE estado_comercial 
        SET precio_publicacion = COALESCE(?, precio_publicacion),
            estado = COALESCE(?, estado),
            condicion = COALESCE(?, condicion),
            origen = COALESCE(?, origen),
            modificado_por = ?
        WHERE id_moto = ?
      `;
      
      await helpers.query(queryEstado, [
        precio_publicacion, estado, condicion, origen, 
        req.user?.userId || req.user?.id, id
      ]);
    }
    
    res.json({
      success: true,
      message: 'Moto actualizada exitosamente'
    });
    
  } catch (error) {
    console.error('Error al actualizar moto:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor'
    });
  }
});

// ==================== ENDPOINTS PARA MÉTRICAS DE BAR-CHART ====================

// Endpoint para métricas de ventas de hoy
router.get('/metricas/ventas-hoy', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_empresa } = req.query;
    
    if (!id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'ID de empresa es requerido'
      });
    }

    const hoy = new Date().toISOString().split('T')[0];
    const ayer = new Date(Date.now() - 24 * 60 * 60 * 1000).toISOString().split('T')[0];

    // Ventas de hoy
    const ventasHoyQuery = `
      SELECT COALESCE(SUM(precio_venta), 0) as total
      FROM ventas 
      WHERE id_empresa = ? 
      AND DATE(fecha_venta) = ?
    `;
    
    // Ventas de ayer
    const ventasAyerQuery = `
      SELECT COALESCE(SUM(precio_venta), 0) as total
      FROM ventas 
      WHERE id_empresa = ? 
      AND DATE(fecha_venta) = ?
    `;

    const [ventasHoy, ventasAyer] = await Promise.all([
      helpers.query(ventasHoyQuery, [id_empresa, hoy]),
      helpers.query(ventasAyerQuery, [id_empresa, ayer])
    ]);

    const totalHoy = ventasHoy[0]?.total || 0;
    const totalAyer = ventasAyer[0]?.total || 0;
    const cambio = totalAyer > 0 ? ((totalHoy - totalAyer) / totalAyer * 100).toFixed(1) : 0;

    res.json({
      success: true,
      total: totalHoy,
      cambio: parseFloat(cambio)
    });
  } catch (error) {
    console.error('Error obteniendo ventas de hoy:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para métricas de motos vendidas
router.get('/metricas/motos-vendidas', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_empresa } = req.query;
    
    if (!id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'ID de empresa es requerido'
      });
    }

    const mesActual = new Date().getMonth() + 1;
    const añoActual = new Date().getFullYear();

    // Total de motos vendidas
    const totalQuery = `
      SELECT COUNT(*) as total
      FROM ventas 
      WHERE id_empresa = ?
    `;
    
    // Motos vendidas este mes
    const esteMesQuery = `
      SELECT COUNT(*) as este_mes
      FROM ventas 
      WHERE id_empresa = ? 
      AND mes_venta = ? 
      AND año_venta = ?
    `;

    const [total, esteMes] = await Promise.all([
      helpers.query(totalQuery, [id_empresa]),
      helpers.query(esteMesQuery, [id_empresa, mesActual, añoActual])
    ]);

    res.json({
      success: true,
      total: total[0]?.total || 0,
      este_mes: esteMes[0]?.este_mes || 0
    });
  } catch (error) {
    console.error('Error obteniendo motos vendidas:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para métricas de inventario
router.get('/metricas/inventario', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_empresa } = req.query;
    
    if (!id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'ID de empresa es requerido'
      });
    }

    // Cantidad de motos disponibles
    const cantidadQuery = `
      SELECT COUNT(*) as cantidad
      FROM motos m
      JOIN estado_comercial ec ON m.id_moto = ec.id_moto
      WHERE m.id_empresa = ? 
      AND ec.estado IN ('DISPONIBLE', 'RESERVADA')
    `;
    
    // Valor total del inventario
    const valorQuery = `
      SELECT COALESCE(SUM(ec.precio_publicacion), 0) as valor_total
      FROM motos m
      JOIN estado_comercial ec ON m.id_moto = ec.id_moto
      WHERE m.id_empresa = ? 
      AND ec.estado IN ('DISPONIBLE', 'RESERVADA')
    `;

    const [cantidad, valor] = await Promise.all([
      helpers.query(cantidadQuery, [id_empresa]),
      helpers.query(valorQuery, [id_empresa])
    ]);

    res.json({
      success: true,
      cantidad: cantidad[0]?.cantidad || 0,
      valor_total: valor[0]?.valor_total || 0
    });
  } catch (error) {
    console.error('Error obteniendo inventario:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para métricas de consignaciones
router.get('/metricas/consignaciones', middleware.authenticateToken, async (req, res) => {
  try {
    const { id_empresa } = req.query;
    
    if (!id_empresa) {
      return res.status(400).json({
        success: false,
        message: 'ID de empresa es requerido'
      });
    }

    // Cantidad de motos en consignación
    const cantidadQuery = `
      SELECT COUNT(*) as cantidad
      FROM motos m
      JOIN estado_comercial ec ON m.id_moto = ec.id_moto
      WHERE m.id_empresa = ? 
      AND ec.origen = 'CONSIGNACION'
      AND ec.estado = 'DISPONIBLE'
    `;
    
    // Valor esperado de consignaciones
    const valorQuery = `
      SELECT COALESCE(SUM(ec.precio_publicacion), 0) as valor_esperado
      FROM motos m
      JOIN estado_comercial ec ON m.id_moto = ec.id_moto
      WHERE m.id_empresa = ? 
      AND ec.origen = 'CONSIGNACION'
      AND ec.estado = 'DISPONIBLE'
    `;

    const [cantidad, valor] = await Promise.all([
      helpers.query(cantidadQuery, [id_empresa]),
      helpers.query(valorQuery, [id_empresa])
    ]);

    res.json({
      success: true,
      cantidad: cantidad[0]?.cantidad || 0,
      valor_esperado: valor[0]?.valor_esperado || 0
    });
  } catch (error) {
    console.error('Error obteniendo consignaciones:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// ==================== ENDPOINTS PARA ANÁLISIS EMPRESARIAL ====================

// Endpoint para análisis de clientes por región
router.get('/analisis/clientes-por-region', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa } = req.query;
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    const query = `
      SELECT 
        r.id as region_id,
        r.nombre as region_nombre,
        r.codigo as region_codigo,
        COUNT(c.id) as total_clientes,
        COUNT(CASE WHEN c.tipo_cliente = 'NATURAL' THEN 1 END) as clientes_naturales,
        COUNT(CASE WHEN c.tipo_cliente = 'EMPRESA' THEN 1 END) as clientes_empresas,
        MIN(c.fecha_creacion) as primer_cliente,
        MAX(c.fecha_creacion) as ultimo_cliente
      FROM regiones r
      LEFT JOIN clientes c ON r.id = c.id_region AND c.id_empresa = ?
      WHERE r.activa = TRUE
      GROUP BY r.id, r.nombre, r.codigo
      HAVING total_clientes > 0
      ORDER BY total_clientes DESC
    `;
    
    const result = await helpers.query(query, [empresa]);
    
    res.json({ 
      success: true,
      data: result,
      empresa: empresa,
      message: `Análisis de clientes por región para empresa ${empresa}`
    });
  } catch (error) {
    console.error('Error en análisis clientes por región:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para análisis de ventas por región
router.get('/analisis/ventas-por-region', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa, periodo } = req.query;
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    let filtroFecha = '';
    if (periodo === 'mes') {
      filtroFecha = 'AND v.fecha_venta >= DATE_SUB(NOW(), INTERVAL 1 MONTH)';
    } else if (periodo === 'trimestre') {
      filtroFecha = 'AND v.fecha_venta >= DATE_SUB(NOW(), INTERVAL 3 MONTH)';
    } else if (periodo === 'año') {
      filtroFecha = 'AND v.fecha_venta >= DATE_SUB(NOW(), INTERVAL 1 YEAR)';
    }

    const query = `
      SELECT 
        r.id as region_id,
        r.nombre as region_nombre,
        COUNT(v.id) as total_ventas,
        SUM(v.precio_venta) as monto_total,
        AVG(v.precio_venta) as ticket_promedio,
        MIN(v.fecha_venta) as primera_venta,
        MAX(v.fecha_venta) as ultima_venta,
        COUNT(DISTINCT c.id) as clientes_unicos,
        SUM(v.descuento) as descuento_total,
        SUM(v.comision) as comision_total
      FROM regiones r
      INNER JOIN clientes c ON r.id = c.id_region AND c.id_empresa = ?
      INNER JOIN ventas v ON c.id = v.id_cliente
      WHERE v.id_empresa = ?
      ${filtroFecha}
      GROUP BY r.id, r.nombre
      ORDER BY monto_total DESC
    `;
    
    const result = await helpers.query(query, [empresa, empresa]);
    
    res.json({ 
      success: true,
      data: result,
      empresa: empresa,
      periodo: periodo || 'todos',
      message: `Análisis de ventas por región para empresa ${empresa}`
    });
  } catch (error) {
    console.error('Error en análisis ventas por región:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para análisis de inventario por origen
router.get('/analisis/inventario-por-origen', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa } = req.query;
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    const query = `
      SELECT 
        ec.origen,
        ec.condicion,
        ec.estado,
        COUNT(m.id_moto) as cantidad_motos,
        SUM(ec.precio_publicacion) as valor_total,
        AVG(ec.precio_publicacion) as precio_promedio,
        MIN(ec.fecha_ingreso) as fecha_primer_ingreso,
        MAX(ec.fecha_ingreso) as fecha_ultimo_ingreso,
        COUNT(CASE WHEN ec.estado = 'DISPONIBLE' THEN 1 END) as disponibles,
        COUNT(CASE WHEN ec.estado = 'VENDIDA' THEN 1 END) as vendidas,
        COUNT(CASE WHEN ec.estado = 'RESERVADA' THEN 1 END) as reservadas
      FROM motos m
      INNER JOIN estado_comercial ec ON m.id_moto = ec.id_moto
      WHERE m.id_empresa = ?
      GROUP BY ec.origen, ec.condicion, ec.estado
      ORDER BY cantidad_motos DESC
    `;
    
    const result = await helpers.query(query, [empresa]);
    
    // Agrupar por origen principal
    const agrupado = {};
    result.forEach(row => {
      if (!agrupado[row.origen]) {
        agrupado[row.origen] = {
          origen: row.origen,
          total_motos: 0,
          valor_total: 0,
          disponibles: 0,
          vendidas: 0,
          reservadas: 0,
          detalles: []
        };
      }
      
      agrupado[row.origen].total_motos += row.cantidad_motos;
      agrupado[row.origen].valor_total += parseFloat(row.valor_total || 0);
      agrupado[row.origen].disponibles += row.disponibles;
      agrupado[row.origen].vendidas += row.vendidas;
      agrupado[row.origen].reservadas += row.reservadas;
      agrupado[row.origen].detalles.push(row);
    });
    
    const resultadoFinal = Object.values(agrupado);
    
    res.json({ 
      success: true,
      data: resultadoFinal,
      detalle: result,
      empresa: empresa,
      message: `Análisis de inventario por origen para empresa ${empresa}`
    });
  } catch (error) {
    console.error('Error en análisis inventario por origen:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para métricas resumidas de empresa
router.get('/analisis/metricas-resumen', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa } = req.query;
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    // Ejecutar múltiples consultas en paralelo
    const [clientesTotal, ventasTotal, inventarioTotal, regionesActivas] = await Promise.all([
      // Total de clientes
      helpers.query('SELECT COUNT(*) as total FROM clientes WHERE id_empresa = ?', [empresa]),
      
      // Total de ventas este mes
      helpers.query(`
        SELECT 
          COUNT(*) as cantidad,
          SUM(precio_venta) as monto_total,
          AVG(precio_venta) as ticket_promedio
        FROM ventas 
        WHERE id_empresa = ? 
        AND fecha_venta >= DATE_SUB(NOW(), INTERVAL 1 MONTH)
      `, [empresa]),
      
      // Total de inventario
      helpers.query(`
        SELECT 
          COUNT(m.id_moto) as cantidad,
          SUM(ec.precio_publicacion) as valor_total
        FROM motos m
        INNER JOIN estado_comercial ec ON m.id_moto = ec.id_moto
        WHERE m.id_empresa = ?
        AND ec.estado IN ('DISPONIBLE', 'RESERVADA')
      `, [empresa]),
      
      // Regiones con clientes
      helpers.query(`
        SELECT COUNT(DISTINCT c.id_region) as regiones_activas
        FROM clientes c
        WHERE c.id_empresa = ?
      `, [empresa])
    ]);

    const metricas = {
      clientes: {
        total: clientesTotal[0]?.total || 0
      },
      ventas: {
        cantidad_mes: ventasTotal[0]?.cantidad || 0,
        monto_total_mes: parseFloat(ventasTotal[0]?.monto_total || 0),
        ticket_promedio: parseFloat(ventasTotal[0]?.ticket_promedio || 0)
      },
      inventario: {
        cantidad_disponible: inventarioTotal[0]?.cantidad || 0,
        valor_total: parseFloat(inventarioTotal[0]?.valor_total || 0)
      },
      regiones: {
        activas: regionesActivas[0]?.regiones_activas || 0
      }
    };
    
    res.json({ 
      success: true,
      data: metricas,
      empresa: empresa,
      fecha_calculo: new Date().toISOString(),
      message: `Métricas resumidas para empresa ${empresa}`
    });
  } catch (error) {
    console.error('Error en métricas resumen:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para análisis comparativo temporal
router.get('/analisis/comparativo-temporal', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa, tipo = 'mensual' } = req.query;
    
    if (!empresa) {
      return res.status(400).json({
        success: false,
        message: 'El parámetro empresa es requerido'
      });
    }

    let agrupacion, filtro;
    
    switch (tipo) {
      case 'semanal':
        agrupacion = 'YEARWEEK(fecha_venta)';
        filtro = 'AND fecha_venta >= DATE_SUB(NOW(), INTERVAL 12 WEEK)';
        break;
      case 'trimestral':
        agrupacion = 'CONCAT(YEAR(fecha_venta), "-Q", QUARTER(fecha_venta))';
        filtro = 'AND fecha_venta >= DATE_SUB(NOW(), INTERVAL 2 YEAR)';
        break;
      case 'anual':
        agrupacion = 'YEAR(fecha_venta)';
        filtro = 'AND fecha_venta >= DATE_SUB(NOW(), INTERVAL 5 YEAR)';
        break;
      default: // mensual
        agrupacion = 'DATE_FORMAT(fecha_venta, "%Y-%m")';
        filtro = 'AND fecha_venta >= DATE_SUB(NOW(), INTERVAL 12 MONTH)';
    }

    const query = `
      SELECT 
        ${agrupacion} as periodo,
        COUNT(v.id) as total_ventas,
        SUM(v.precio_venta) as monto_total,
        AVG(v.precio_venta) as ticket_promedio,
        COUNT(DISTINCT v.id_cliente) as clientes_unicos,
        COUNT(DISTINCT c.id_region) as regiones_vendidas
      FROM ventas v
      INNER JOIN clientes c ON v.id_cliente = c.id
      WHERE v.id_empresa = ?
      ${filtro}
      GROUP BY ${agrupacion}
      ORDER BY periodo ASC
    `;
    
    const result = await helpers.query(query, [empresa]);
    
    res.json({ 
      success: true,
      data: result,
      empresa: empresa,
      tipo_analisis: tipo,
      message: `Análisis comparativo ${tipo} para empresa ${empresa}`
    });
  } catch (error) {
    console.error('Error en análisis comparativo temporal:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para subir imágenes de motos
router.post('/motos/:id/imagenes', middleware.authenticateToken, upload.array('imagenes', 10), async (req, res) => {
  try {
    const motoId = req.params.id;
    const { es_principal } = req.body;
    
    console.log(`📸 Subiendo imágenes para moto ${motoId}:`, {
      archivos: req.files?.length || 0,
      es_principal: es_principal
    });
    
    if (!req.files || req.files.length === 0) {
      return res.status(400).json({
        success: false,
        message: 'No se han subido archivos'
      });
    }

    // Verificar que la moto existe
    const motoExists = await helpers.query(`
      SELECT m.id_moto, m.id_empresa 
      FROM motos m 
      WHERE m.id_moto = ?
    `, [motoId]);

    if (motoExists.length === 0) {
      // Eliminar archivos subidos si la moto no existe
      req.files.forEach(file => {
        if (fs.existsSync(file.path)) {
          fs.unlinkSync(file.path);
        }
      });
      return res.status(404).json({
        success: false,
        message: 'Moto no encontrada'
      });
    }

    // Si se marca como principal, desmarcar todas las demás imágenes principales
    if (es_principal === 'true') {
      await helpers.query(`
        UPDATE imagenes_motos 
        SET es_principal = 0 
        WHERE id_moto = ?
      `, [motoId]);
    }

    const imagenesGuardadas = [];
    
    // Guardar cada imagen en la base de datos
    for (let i = 0; i < req.files.length; i++) {
      const file = req.files[i];
      const esPrincipalImg = (es_principal === 'true' && i === 0) ? 1 : 0;
      
      console.log(`💾 Guardando imagen ${i + 1}:`, {
        filename: file.filename,
        path: file.path,
        es_principal: esPrincipalImg
      });
      
      // Calcular la ruta relativa desde 'uploads' para la base de datos
      const currentDate = new Date();
      const year = currentDate.getFullYear();
      const month = String(currentDate.getMonth() + 1).padStart(2, '0');
      const rutaRelativa = `/uploads/motos/${year}/${month}/${file.filename}`;
      
      const resultado = await helpers.query(`
        INSERT INTO imagenes_motos (id_moto, nombre_archivo, ruta_archivo, es_principal)
        VALUES (?, ?, ?, ?)
      `, [motoId, file.filename, rutaRelativa, esPrincipalImg]);

      // Para helpers.query, el insertId está en resultado.insertId si existe
      const insertId = resultado.insertId || resultado.id || (resultado.length > 0 ? resultado[0].insertId : null);
      
      imagenesGuardadas.push({
        id: insertId,
        nombre_archivo: file.filename,
        ruta_archivo: rutaRelativa,
        es_principal: Boolean(esPrincipalImg),
        size: file.size,
        mimetype: file.mimetype
      });
    }
    
    console.log(`✅ ${req.files.length} imagen(es) guardada(s) exitosamente`);
    
    res.json({
      success: true,
      message: `${req.files.length} imagen(es) subida(s) exitosamente`,
      data: {
        moto_id: motoId,
        imagenes: imagenesGuardadas,
        total_subidas: req.files.length
      }
    });
    
  } catch (error) {
    console.error('❌ Error al subir imágenes:', error);
    
    // Eliminar archivos subidos en caso de error
    if (req.files) {
      req.files.forEach(file => {
        if (fs.existsSync(file.path)) {
          fs.unlinkSync(file.path);
        }
      });
    }
    
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para obtener imágenes de una moto
router.get('/motos/:id/imagenes', middleware.authenticateToken, async (req, res) => {
  try {
    const motoId = req.params.id;
    
    const imagenes = await helpers.query(`
      SELECT 
        id,
        nombre_archivo,
        ruta_archivo,
        es_principal
      FROM imagenes_motos 
      WHERE id_moto = ?
      ORDER BY es_principal DESC, id ASC
    `, [motoId]);

    // Convertir rutas de archivos a URLs accesibles
    const imagenesConUrl = imagenes.map(img => {
      let rutaFinal;
      
      // Si ruta_archivo contiene subcarpetas, usar esa ruta
      if (img.ruta_archivo && img.ruta_archivo.includes('/')) {
        rutaFinal = img.ruta_archivo.replace(/\\/g, '/');
      }
      // Si ruta_archivo es solo el nombre del archivo, usar directamente
      else if (img.ruta_archivo && !img.ruta_archivo.includes('/')) {
        rutaFinal = img.ruta_archivo;
      }
      // Fallback: usar nombre_archivo si no hay ruta_archivo
      else {
        rutaFinal = img.nombre_archivo;
      }
      
      return {
        ...img,
        url: `/uploads/${rutaFinal}`,
        es_principal: Boolean(img.es_principal)
      };
    });

    res.json({
      success: true,
      data: imagenesConUrl,
      total: imagenes.length
    });
    
  } catch (error) {
    console.error('Error al obtener imágenes:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// Endpoint para eliminar una imagen
router.delete('/motos/:motoId/imagenes/:imagenId', middleware.authenticateToken, async (req, res) => {
  try {
    const { motoId, imagenId } = req.params;
    
    // Obtener información de la imagen
    const imagen = await helpers.query(`
      SELECT * FROM imagenes_motos 
      WHERE id = ? AND id_moto = ?
    `, [imagenId, motoId]);

    if (imagen.length === 0) {
      return res.status(404).json({
        success: false,
        message: 'Imagen no encontrada'
      });
    }

    // Eliminar archivo físico
    const rutaArchivo = imagen[0].ruta_archivo;
    if (fs.existsSync(rutaArchivo)) {
      fs.unlinkSync(rutaArchivo);
    }

    // Eliminar registro de la base de datos
    await helpers.query(`
      DELETE FROM imagenes_motos 
      WHERE id = ? AND id_moto = ?
    `, [imagenId, motoId]);

    res.json({
      success: true,
      message: 'Imagen eliminada exitosamente'
    });
    
  } catch (error) {
    console.error('Error al eliminar imagen:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// ==================== ENDPOINT PARA ACTUALIZAR ESTADO DE MOTO ====================

// Endpoint para actualizar el estado de una moto
router.post('/motos/estado', middleware.authenticateToken, async (req, res) => {
  try {
    const { id, estado } = req.body;
    
    console.log('🔄 Actualizando estado de moto:', { id, estado });
    console.log('👤 Usuario autenticado:', { 
      userId: req.user?.userId, 
      id_usuario: req.user?.id_usuario,
      id: req.user?.id,
      correo: req.user?.correo 
    });
    
    // Validar entrada
    if (!id || !estado) {
      return res.status(400).json({
        success: false,
        message: 'ID de moto y estado son requeridos'
      });
    }
    
    // Validar que el estado sea válido
    const estadosValidos = ['DISPONIBLE', 'RESERVADA', 'VENDIDA', 'EN_MANTENIMIENTO'];
    if (!estadosValidos.includes(estado)) {
      return res.status(400).json({
        success: false,
        message: `Estado no válido. Debe ser uno de: ${estadosValidos.join(', ')}`
      });
    }
    
    // Verificar que la moto existe y obtener su estado comercial actual
    console.log('🔍 Buscando moto con ID:', id);
    const estadoComercialExiste = await helpers.query(
      'SELECT ec.id, ec.id_moto, ec.estado, m.id_moto FROM estado_comercial ec INNER JOIN motos m ON ec.id_moto = m.id_moto WHERE m.id_moto = ?',
      [id]
    );
    
    console.log('📊 Resultado de búsqueda:', estadoComercialExiste);
    
    if (!estadoComercialExiste || estadoComercialExiste.length === 0) {
      console.log('❌ Moto no encontrada en estado_comercial');
      return res.status(404).json({
        success: false,
        message: 'Moto no encontrada o sin estado comercial'
      });
    }
    
    const estadoAnterior = estadoComercialExiste[0].estado;
    const usuarioModificador = req.user?.userId || req.user?.id_usuario || req.user?.id || 1;
    const fechaActual = new Date().toISOString().split('T')[0]; // Formato YYYY-MM-DD
    
    console.log('🔄 Preparando actualización:', {
      estado_anterior: estadoAnterior,
      estado_nuevo: estado,
      usuario_modificador: usuarioModificador,
      fecha_toma: fechaActual
    });
    
    // Si el estado es RESERVADA o VENDIDA, también actualizamos id_toma y fecha_toma
    let query, params;
    if (estado === 'RESERVADA' || estado === 'VENDIDA') {
      query = 'UPDATE estado_comercial SET estado = ?, id_toma = ?, fecha_toma = ?, modificado_por = ? WHERE id_moto = ?';
      params = [estado, usuarioModificador, fechaActual, usuarioModificador, id];
      console.log('📝 Actualizando con id_toma y fecha_toma para estado:', estado);
    } else {
      // Para otros estados, solo actualizar estado y modificado_por
      query = 'UPDATE estado_comercial SET estado = ?, modificado_por = ? WHERE id_moto = ?';
      params = [estado, usuarioModificador, id];
      console.log('📝 Actualizando solo estado para:', estado);
    }
    
    // Actualizar el estado en la tabla estado_comercial
    const resultado = await helpers.query(query, params);
    
    console.log('📊 Resultado de actualización:', resultado);
    
    if (resultado.affectedRows === 0) {
      return res.status(500).json({
        success: false,
        message: 'No se pudo actualizar el estado de la moto'
      });
    }
    
    console.log('✅ Estado de moto actualizado exitosamente:', {
      id,
      estadoAnterior,
      estadoNuevo: estado
    });
    
    res.json({
      success: true,
      message: 'Estado de moto actualizado exitosamente',
      data: {
        id_moto: id,
        estado_anterior: estadoAnterior,
        estado_nuevo: estado,
        fecha_actualizacion: new Date()
      }
    });
    
  } catch (error) {
    console.error('❌ Error al actualizar estado de moto:', error);
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor',
      error: error.message
    });
  }
});

// ==================== ENDPOINTS PARA DOCUMENTOS ====================

// Endpoint para guardar documento desde buffer (para PDFs generados en frontend)
router.post('/documentos/guardar', middleware.authenticateToken, async (req, res) => {
  try {
    const {
      tipo_entidad,
      id_entidad,
      tipo_documento,
      nombre_archivo,
      contenido_base64,
      descripcion,
      es_publico = 0
    } = req.body;

    // Validaciones
    if (!tipo_entidad || !id_entidad || !tipo_documento || !nombre_archivo || !contenido_base64) {
      return res.status(400).json({
        success: false,
        message: 'Faltan campos obligatorios'
      });
    }

    // Crear directorio si no existe
    const year = new Date().getFullYear();
    const month = String(new Date().getMonth() + 1).padStart(2, '0');
    const uploadDir = path.join(__dirname, 'uploads', 'documentos', year.toString(), month);
    
    if (!fs.existsSync(uploadDir)){
      fs.mkdirSync(uploadDir, { recursive: true });
    }

    // Generar nombre único para el archivo
    const timestamp = Date.now();
    const nombreUnico = `${timestamp}_${nombre_archivo}`;
    const rutaCompleta = path.join(uploadDir, nombreUnico);
    const rutaRelativa = path.join('documentos', year.toString(), month, nombreUnico);

    // Convertir base64 a buffer y guardar archivo
    const buffer = Buffer.from(contenido_base64, 'base64');
    fs.writeFileSync(rutaCompleta, buffer);

    // Generar hash del archivo
    const hash = await generateFileHash(rutaCompleta);

    // Guardar en base de datos
    const connection = await pool.promise().getConnection();
    try {
      const documentoId = await guardarDocumentoBD(connection, {
        tipo_entidad,
        id_entidad,
        tipo_documento,
        nombre_archivo: nombreUnico,
        ruta_archivo: rutaRelativa,
        tamaño_archivo: buffer.length,
        mime_type: 'application/pdf',
        hash_archivo: hash,
        descripcion,
        es_publico,
        creado_por: req.user.id
      });

      res.json({
        success: true,
        message: 'Documento guardado exitosamente',
        data: {
          id: documentoId,
          nombre_archivo: nombreUnico,
          ruta_archivo: rutaRelativa,
          tamaño: buffer.length
        }
      });

    } finally {
      connection.release();
    }

  } catch (error) {
    console.error('Error guardando documento:', error);
    res.status(500).json({
      success: false,
      message: 'Error al guardar documento',
      error: error.message
    });
  }
});

// Endpoint para subir documento vía upload
router.post('/documentos/upload', middleware.authenticateToken, uploadDocument.single('documento'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({
        success: false,
        message: 'No se recibió ningún archivo'
      });
    }

    const {
      tipo_entidad,
      id_entidad,
      tipo_documento,
      descripcion,
      es_publico = 0
    } = req.body;

    // Validaciones
    if (!tipo_entidad || !id_entidad || !tipo_documento) {
      // Eliminar archivo subido si faltan datos
      fs.unlinkSync(req.file.path);
      return res.status(400).json({
        success: false,
        message: 'Faltan campos obligatorios'
      });
    }

    // Generar hash del archivo
    const hash = await generateFileHash(req.file.path);

    // Obtener ruta relativa
    const rutaRelativa = path.relative(path.join(__dirname, 'uploads'), req.file.path);

    // Guardar en base de datos
    let connection;
    try {
      connection = await pool.promise().getConnection();
      const documentoId = await guardarDocumentoBD(connection, {
        tipo_entidad,
        id_entidad,
        tipo_documento,
        nombre_archivo: req.file.filename,
        ruta_archivo: rutaRelativa,
        tamaño_archivo: req.file.size,
        mime_type: req.file.mimetype,
        hash_archivo: hash,
        descripcion,
        es_publico,
        creado_por: req.user.userId || req.user.id
      });

      res.json({
        success: true,
        message: 'Documento subido exitosamente',
        data: {
          id: documentoId,
          nombre_archivo: req.file.filename,
          ruta_archivo: rutaRelativa,
          tamaño: req.file.size
        }
      });

    } finally {
      if (connection) {
        connection.release();
      }
    }

  } catch (error) {
    console.error('Error subiendo documento:', error);
    // Limpiar archivo si hay error
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path);
    }
    
    res.status(500).json({
      success: false,
      message: 'Error al subir documento',
      error: error.message
    });
  }
});

// Endpoint POST simplificado para documentos (usar multer de documentos)
router.post('/documentos', middleware.authenticateToken, uploadDocument.single('archivo'), async (req, res) => {
  try {
    console.log('📄 POST /documentos - Inicio del endpoint');
    console.log('📄 User desde token:', req.user);
    console.log('📄 Body recibido:', req.body);
    console.log('📄 Archivo recibido:', req.file ? req.file.filename : 'No hay archivo');

    if (!req.file) {
      console.log('❌ No se recibió ningún archivo');
      return res.status(400).json({
        success: false,
        message: 'No se recibió ningún archivo'
      });
    }

    const {
      tipo_entidad,
      id_entidad,
      tipo_documento,
      descripcion,
      es_publico = 0
    } = req.body;

    console.log('📄 Parámetros extraídos:', {
      tipo_entidad,
      id_entidad,
      tipo_documento,
      descripcion,
      es_publico
    });

    // Validaciones
    if (!tipo_entidad || !id_entidad || !tipo_documento) {
      console.log('❌ Faltan campos obligatorios');
      // Eliminar archivo subido si faltan datos
      fs.unlinkSync(req.file.path);
      return res.status(400).json({
        success: false,
        message: 'Faltan campos obligatorios: tipo_entidad, id_entidad, tipo_documento'
      });
    }

    // Generar hash del archivo
    console.log('📄 Generando hash del archivo...');
    const hash = await generateFileHash(req.file.path);
    console.log('📄 Hash generado:', hash);

    // Obtener ruta relativa
    const rutaRelativa = path.relative(path.join(__dirname, 'uploads'), req.file.path);
    console.log('📄 Ruta relativa:', rutaRelativa);

    // Guardar en base de datos
    let connection;
    try {
      console.log('📄 Obteniendo conexión a BD...');
      connection = await pool.promise().getConnection();
      
      console.log('📄 Guardando documento en BD...');
      const documentoId = await guardarDocumentoBD(connection, {
        tipo_entidad,
        id_entidad,
        tipo_documento,
        nombre_archivo: req.file.filename,
        ruta_archivo: rutaRelativa,
        tamaño_archivo: req.file.size,
        mime_type: req.file.mimetype,
        hash_archivo: hash,
        descripcion: descripcion || `Documento ${tipo_documento}`,
        es_publico,
        creado_por: req.user.id
      });

      console.log('✅ Documento guardado con ID:', documentoId);

      res.json({
        success: true,
        message: 'Documento subido exitosamente',
        data: {
          id: documentoId,
          nombre_archivo: req.file.filename,
          ruta_archivo: rutaRelativa,
          tamaño: req.file.size,
          tipo_documento
        }
      });

    } finally {
      if (connection) {
        connection.release();
      }
    }

  } catch (error) {
    console.error('❌ Error subiendo documento:', error);
    console.error('❌ Stack trace:', error.stack);
    // Limpiar archivo si hay error
    if (req.file && fs.existsSync(req.file.path)) {
      console.log('🗑️ Eliminando archivo por error:', req.file.path);
      fs.unlinkSync(req.file.path);
    }
    
    res.status(500).json({
      success: false,
      message: 'Error al subir documento',
      error: error.message
    });
  }
});

// Endpoint para obtener documentos de una entidad
router.get('/documentos/:tipo_entidad/:id_entidad', middleware.authenticateToken, async (req, res) => {
  try {
    const { tipo_entidad, id_entidad } = req.params;
    const { tipo_documento } = req.query;

    let query = `
      SELECT 
        d.*,
        u.nombre as creador_nombre
      FROM documentos d
      LEFT JOIN usuarios u ON d.creado_por = u.id
      WHERE d.tipo_entidad = ? AND d.id_entidad = ?
    `;
    
    const params = [tipo_entidad, id_entidad];

    if (tipo_documento) {
      query += ' AND d.tipo_documento = ?';
      params.push(tipo_documento);
    }

    query += ' ORDER BY d.fecha_creacion DESC';

    const documentos = await helpers.query(query, params);

    res.json({
      success: true,
      data: documentos
    });

  } catch (error) {
    console.error('Error obteniendo documentos:', error);
    res.status(500).json({
      success: false,
      message: 'Error al obtener documentos',
      error: error.message
    });
  }
});

// Endpoint para descargar documento
router.get('/documentos/descargar/:id', middleware.authenticateToken, async (req, res) => {
  try {
    const documentoId = req.params.id;

    const [documento] = await helpers.query(
      'SELECT * FROM documentos WHERE id = ?',
      [documentoId]
    );

    if (!documento) {
      return res.status(404).json({
        success: false,
        message: 'Documento no encontrado'
      });
    }

    const rutaCompleta = path.join(__dirname, 'uploads', documento.ruta_archivo);

    if (!fs.existsSync(rutaCompleta)) {
      return res.status(404).json({
        success: false,
        message: 'Archivo físico no encontrado'
      });
    }

    res.setHeader('Content-Type', documento.mime_type);
    res.setHeader('Content-Disposition', `attachment; filename="${documento.nombre_archivo}"`);
    
    const fileStream = fs.createReadStream(rutaCompleta);
    fileStream.pipe(res);

  } catch (error) {
    console.error('Error descargando documento:', error);
    res.status(500).json({
      success: false,
      message: 'Error al descargar documento',
      error: error.message
    });
  }
});

// Endpoint para eliminar documento
router.delete('/documentos/:id', middleware.authenticateToken, async (req, res) => {
  try {
    const documentoId = req.params.id;

    // Obtener información del documento antes de eliminarlo
    const [documento] = await helpers.query(
      'SELECT * FROM documentos WHERE id = ?',
      [documentoId]
    );

    if (!documento) {
      return res.status(404).json({
        success: false,
        message: 'Documento no encontrado'
      });
    }

    // Verificar permisos (solo el creador o admin puede eliminar)
    if (documento.creado_por !== req.user.id && req.user.rol !== 'ADMIN') {
      return res.status(403).json({
        success: false,
        message: 'No tienes permisos para eliminar este documento'
      });
    }

    const connection = await pool.promise().getConnection();
    try {
      await connection.beginTransaction();

      // Eliminar registro de la base de datos
      await connection.execute(
        'DELETE FROM documentos WHERE id = ?',
        [documentoId]
      );

      // Eliminar archivo físico
      const rutaCompleta = path.join(__dirname, 'uploads', documento.ruta_archivo);
      if (fs.existsSync(rutaCompleta)) {
        fs.unlinkSync(rutaCompleta);
      }

      await connection.commit();

      res.json({
        success: true,
        message: 'Documento eliminado exitosamente'
      });

    } catch (error) {
      await connection.rollback();
      throw error;
    } finally {
      connection.release();
    }

  } catch (error) {
    console.error('Error eliminando documento:', error);
    res.status(500).json({
      success: false,
      message: 'Error al eliminar documento',
      error: error.message
    });
  }
});

// Endpoint para actualizar metadatos de documento
router.put('/documentos/:id', middleware.authenticateToken, async (req, res) => {
  try {
    const documentoId = req.params.id;
    const { descripcion, es_publico } = req.body;

    // Verificar que el documento existe
    const [documento] = await helpers.query(
      'SELECT * FROM documentos WHERE id = ?',
      [documentoId]
    );

    if (!documento) {
      return res.status(404).json({
        success: false,
        message: 'Documento no encontrado'
      });
    }

    // Verificar permisos
    if (documento.creado_por !== req.user.id && req.user.rol !== 'ADMIN') {
      return res.status(403).json({
        success: false,
        message: 'No tienes permisos para modificar este documento'
      });
    }

    // Actualizar metadatos
    await helpers.query(`
      UPDATE documentos 
      SET descripcion = ?, es_publico = ?, fecha_actualizacion = NOW()
      WHERE id = ?
    `, [descripcion, es_publico, documentoId]);

    res.json({
      success: true,
      message: 'Documento actualizado exitosamente'
    });

  } catch (error) {
    console.error('Error actualizando documento:', error);
    res.status(500).json({
      success: false,
      message: 'Error al actualizar documento',
      error: error.message
    });
  }
});

// Endpoint para obtener documentos de una venta específica
router.get('/ventas/:id/documentos', middleware.authenticateToken, async (req, res) => {
  try {
    const ventaId = req.params.id;

    // Verificar que la venta existe
    const [venta] = await helpers.query(
      'SELECT * FROM ventas WHERE id = ?',
      [ventaId]
    );

    if (!venta) {
      return res.status(404).json({
        success: false,
        message: 'Venta no encontrada'
      });
    }

    // Obtener documentos de la venta
    const documentos = await helpers.query(`
      SELECT 
        d.*,
        u.nombre as creador_nombre
      FROM documentos d
      LEFT JOIN usuarios u ON d.creado_por = u.id
      WHERE d.tipo_entidad = 'VENTA' AND d.id_entidad = ?
      ORDER BY d.fecha_creacion DESC
    `, [ventaId]);

    res.json({
      success: true,
      data: {
        venta: {
          id: venta.id,
          fecha_venta: venta.fecha_venta,
          precio_venta: venta.precio_venta
        },
        documentos: documentos
      }
    });

  } catch (error) {
    console.error('Error obteniendo documentos de venta:', error);
    res.status(500).json({
      success: false,
      message: 'Error al obtener documentos de la venta',
      error: error.message
    });
  }
});

// Endpoint para verificar integridad de documento
router.get('/documentos/:id/verificar', middleware.authenticateToken, async (req, res) => {
  try {
    const documentoId = req.params.id;

    const [documento] = await helpers.query(
      'SELECT * FROM documentos WHERE id = ?',
      [documentoId]
    );

    if (!documento) {
      return res.status(404).json({
        success: false,
        message: 'Documento no encontrado'
      });
    }

    const rutaCompleta = path.join(__dirname, 'uploads', documento.ruta_archivo);

    if (!fs.existsSync(rutaCompleta)) {
      return res.json({
        success: true,
        data: {
          existe_archivo: false,
          integridad_ok: false,
          mensaje: 'Archivo físico no encontrado'
        }
      });
    }

    // Verificar hash del archivo
    const hashActual = await generateFileHash(rutaCompleta);
    const integridadOK = hashActual === documento.hash_archivo;

    res.json({
      success: true,
      data: {
        existe_archivo: true,
        integridad_ok: integridadOK,
        hash_bd: documento.hash_archivo,
        hash_actual: hashActual,
        mensaje: integridadOK ? 'Documento íntegro' : 'Documento modificado o corrupto'
      }
    });

  } catch (error) {
    console.error('Error verificando documento:', error);
    res.status(500).json({
      success: false,
      message: 'Error al verificar documento',
      error: error.message
    });
  }
});

// Endpoint para obtener estadísticas de documentos
router.get('/documentos/estadisticas', middleware.authenticateToken, async (req, res) => {
  try {
    const { empresa } = req.query;
    
    let whereClause = '';
    let params = [];
    
    if (empresa) {
      // Para filtrar por empresa, necesitamos hacer joins según el tipo de entidad
      whereClause = `
        WHERE (
          (d.tipo_entidad = 'VENTA' AND EXISTS (SELECT 1 FROM ventas v WHERE v.id = d.id_entidad AND v.id_empresa = ?)) OR
          (d.tipo_entidad = 'MOTO' AND EXISTS (SELECT 1 FROM motos m WHERE m.id_moto = d.id_entidad AND m.id_empresa = ?)) OR
          (d.tipo_entidad = 'CLIENTE' AND EXISTS (SELECT 1 FROM clientes c WHERE c.id = d.id_entidad AND c.id_empresa = ?)) OR
          (d.tipo_entidad = 'EMPRESA' AND d.id_entidad = ?)
        )
      `;
      params = [empresa, empresa, empresa, empresa];
    }

    const estadisticas = await helpers.query(`
      SELECT 
        tipo_entidad,
        tipo_documento,
        COUNT(*) as cantidad,
        SUM(tamaño_archivo) as tamaño_total,
        AVG(tamaño_archivo) as tamaño_promedio
      FROM documentos d
      ${whereClause}
      GROUP BY tipo_entidad, tipo_documento
      ORDER BY tipo_entidad, cantidad DESC
    `, params);

    const resumen = await helpers.query(`
      SELECT 
        COUNT(*) as total_documentos,
        SUM(tamaño_archivo) as tamaño_total_bytes,
        ROUND(SUM(tamaño_archivo) / 1024 / 1024, 2) as tamaño_total_mb
      FROM documentos d
      ${whereClause}
    `, params);

    res.json({
      success: true,
      data: {
        resumen: resumen[0],
        por_tipo: estadisticas
      }
    });

  } catch (error) {
    console.error('Error obteniendo estadísticas:', error);
    res.status(500).json({
      success: false,
      message: 'Error al obtener estadísticas de documentos',
      error: error.message
    });
  }
});

// Endpoint para listar documentos (para pruebas de desarrollo)
router.get('/documentos/list', middleware.authenticateToken, async (req, res) => {
  try {
    console.log('📋 Listando documentos...');
    
    const documentos = await helpers.query(`
      SELECT 
        id,
        numero_documento,
        tipo_documento,
        tipo_entidad,
        id_entidad,
        id_cliente,
        id_moto,
        ruta_archivo,
        tamaño_archivo,
        fecha_creacion,
        permisos
      FROM documentos 
      ORDER BY fecha_creacion DESC 
      LIMIT 50
    `);
    
    console.log(`✅ ${documentos.length} documentos encontrados`);
    
    res.json({
      success: true,
      message: `${documentos.length} documentos encontrados`,
      data: documentos
    });
    
  } catch (error) {
    console.error('❌ Error listando documentos:', error);
    res.status(500).json({
      success: false,
      message: 'Error al listar documentos',
      error: error.message
    });
  }
});

module.exports = router;
