const express = require('express');
const cors = require('cors');
const path = require('path');
const fs = require('fs');
require('dotenv').config();

const app = express();

// Configuración del puerto
const PORT = process.env.PORT || 3001;

// CORS configurado para producción
const corsOptions = {
  origin: process.env.NODE_ENV === 'production' 
    ? ['https://admin.bbcrono.airr.cl', 'https://bbcrono.airr.cl']
    : ['http://localhost:3000', 'http://127.0.0.1:3000', 'http://localhost:8080', 'http://localhost:5500', 'http://127.0.0.1:5500'],
  credentials: true,
  optionsSuccessStatus: 200,
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS', 'PATCH'],
  allowedHeaders: [
    'Origin',
    'X-Requested-With',
    'Content-Type',
    'Accept',
    'Authorization',
    'Cache-Control',
    'x-user-role'
  ],
  exposedHeaders: ['Content-Range', 'X-Content-Range', 'Authorization'],
  preflightContinue: false,
  maxAge: 600 // 10 minutos
};

// Aplicar CORS antes que cualquier otro middleware
app.use(cors(corsOptions));

// Middleware de seguridad
const helmet = require('helmet');
app.use(helmet({
  contentSecurityPolicy: false, // Deshabilitado para desarrollo
  crossOriginEmbedderPolicy: false
}));

const routes = require('./routes');
const trabajadoresRoutes = require('./routes-trabajadores');

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({ success: true, message: 'Server is healthy' });
});

// Test endpoint para verificar archivos
app.get('/test-files', (req, res) => {
  const operacionesDir = path.join(__dirname, '../src/operaciones');
  const files = fs.readdirSync(operacionesDir, { withFileTypes: true });
  
  const fileList = files.map(file => ({
    name: file.name,
    isFile: file.isFile(),
    isDirectory: file.isDirectory(),
    fullPath: path.join(operacionesDir, file.name)
  }));
  
  res.json({
    success: true,
    operacionesDirectory: operacionesDir,
    files: fileList,
    srcDirectory: path.join(__dirname, '../src'),
    message: 'Lista de archivos en directorio operaciones'
  });
});

// Middleware básico
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Manejar preflight requests explícitamente
app.options('*', cors(corsOptions));

// Logging middleware con más detalles
app.use((req, res, next) => {
  const timestamp = new Date().toISOString();
  const origin = req.get('Origin') || 'no-origin';
  console.log(`${timestamp} - ${req.method} ${req.url} - Origin: ${origin}`);
  
  // Log específico para archivos HTML
  if (req.url.endsWith('.html')) {
    console.log(`🌐 Solicitando página HTML: ${req.url}`);
  }
  
  // Log CORS headers en respuesta
  res.on('finish', () => {
    const corsHeader = res.get('Access-Control-Allow-Origin');
    if (req.method === 'OPTIONS' || req.url.includes('/api/')) {
      console.log(`  → Response: ${res.statusCode} - CORS Origin: ${corsHeader || 'not-set'}`);
    }
  });
  
  next();
});

// API routes con todas las funcionalidades completas
app.use('/api', routes);
app.use('/api/trabajadores', trabajadoresRoutes);

// Servir archivos de uploads (imágenes)
app.use('/uploads', express.static(path.join(__dirname, 'uploads'), {
  setHeaders: (res, path) => {
    res.set('Cache-Control', 'public, max-age=86400'); // Cache por 1 día
  }
}));

// Servir archivos estáticos con configuración extendida
const staticPath = process.env.NODE_ENV === 'production' 
  ? path.join(__dirname, '../build')
  : path.join(__dirname, '../src');

console.log(`📁 Sirviendo archivos estáticos desde: ${staticPath}`);

app.use(express.static(staticPath, {
  extensions: ['html', 'htm'],
  setHeaders: (res, path) => {
    if (process.env.NODE_ENV === 'production') {
      res.set('Cache-Control', 'public, max-age=3600'); // Cache 1 hora en producción
    } else {
      res.set('Cache-Control', 'no-cache');
    }
  }
}));

// Logging para debugging de archivos estáticos
app.use((req, res, next) => {
  if (!req.url.startsWith('/api')) {
    console.log(`📄 Solicitando archivo: ${req.url}`);
    const staticPath = process.env.NODE_ENV === 'production' 
      ? path.join(__dirname, '../build')
      : path.join(__dirname, '../src');
    const filePath = path.join(staticPath, req.url);
    console.log(`📁 Ruta completa: ${filePath}`);
    console.log(`🔍 Archivo existe: ${fs.existsSync(filePath)}`);
  }
  next();
});

// Ruta específica para operaciones
app.get('/operaciones/*', (req, res) => {
  const fileName = path.basename(req.path);
  const staticPath = process.env.NODE_ENV === 'production' 
    ? path.join(__dirname, '../build')
    : path.join(__dirname, '../src');
  const filePath = path.join(staticPath, 'operaciones', fileName);
  
  console.log(`🎯 Ruta de operaciones solicitada: ${req.path}`);
  console.log(`📂 Buscando archivo en: ${filePath}`);
  console.log(`✅ Archivo existe: ${fs.existsSync(filePath)}`);
  
  if (fs.existsSync(filePath)) {
    res.sendFile(filePath);
  } else {
    console.log(`❌ Archivo no encontrado: ${filePath}`);
    res.status(404).json({ error: 'Archivo no encontrado' });
  }
});

// Ruta por defecto para SPA
app.get('*', (req, res) => {
  // Si no es una ruta de API, servir index.html
  if (!req.url.startsWith('/api')) {
    const staticPath = process.env.NODE_ENV === 'production' 
      ? path.join(__dirname, '../build')
      : path.join(__dirname, '../src');
    res.sendFile(path.join(staticPath, 'index.html'));
  } else {
    res.status(404).json({ error: 'API endpoint not found' });
  }
});

// Manejo de errores global
app.use((err, req, res, next) => {
  console.error('Error del servidor:', err);
  
  if (err.code === 'EADDRINUSE') {
    console.error(`❌ El puerto ${PORT} ya está en uso. Prueba con otro puerto.`);
    process.exit(1);
  }
  
  res.status(500).json({
    success: false,
    error: process.env.NODE_ENV === 'production' 
      ? 'Error interno del servidor' 
      : err.message
  });
});

// Iniciar servidor con manejo de errores
const server = app.listen(PORT, '0.0.0.0', (err) => {
  if (err) {
    console.error('❌ Error al iniciar el servidor:', err);
    process.exit(1);
  }
  console.log(`🚀 Servidor ejecutándose en http://localhost:${PORT}`);
  const staticPath = process.env.NODE_ENV === 'production' 
    ? path.join(__dirname, '../build')
    : path.join(__dirname, '../src');
  console.log(`📁 Sirviendo archivos desde: ${staticPath}`);
  console.log(`🌍 Entorno: ${process.env.NODE_ENV || 'development'}`);
});

// Manejo de cierre graceful
process.on('SIGTERM', () => {
  console.log('🛑 Cerrando servidor...');
  server.close(() => {
    console.log('✅ Servidor cerrado correctamente');
    process.exit(0);
  });
});

process.on('SIGINT', () => {
  console.log('🛑 Cerrando servidor...');
  server.close(() => {
    console.log('✅ Servidor cerrado correctamente');
    process.exit(0);
  });
});

// Capturar errores del puerto
server.on('error', (err) => {
  if (err.code === 'EADDRINUSE') {
    console.error(`❌ Error: El puerto ${PORT} ya está en uso.`);
    console.log('💡 Soluciones:');
    console.log(`   1. Usar otro puerto: PORT=3002 npm run server`);
    console.log(`   2. Matar el proceso que usa el puerto: netstat -ano | findstr :${PORT}`);
    console.log(`   3. Cambiar el puerto en el archivo .env`);
    process.exit(1);
  } else {
    console.error('❌ Error del servidor:', err);
    process.exit(1);
  }
});

module.exports = app;
