// Validaciones para formato chileno
// RUT y teléfonos de Chile

/**
 * Valida formato de RUT chileno (XX.XXX.XXX-X)
 * @param {string} rut - RUT a validar
 * @returns {boolean} - true si es válido
 */
function validarRutChileno(rut) {
  if (!rut) return false;
  
  // Remover puntos y guión para validación
  const cleanRut = rut.replace(/[.-]/g, '');
  
  // Debe tener entre 8 y 9 caracteres
  if (cleanRut.length < 8 || cleanRut.length > 9) return false;
  
  const rutDigits = cleanRut.slice(0, -1);
  const verifier = cleanRut.slice(-1).toLowerCase();
  
  // Validar que los primeros dígitos sean números
  if (!/^\d+$/.test(rutDigits)) return false;
  
  // Calcular dígito verificador
  let suma = 0;
  let multiplicador = 2;
  
  for (let i = rutDigits.length - 1; i >= 0; i--) {
    suma += parseInt(rutDigits[i]) * multiplicador;
    multiplicador = multiplicador === 7 ? 2 : multiplicador + 1;
  }
  
  const resto = suma % 11;
  const dv = 11 - resto;
  const dvCalculado = dv === 11 ? '0' : dv === 10 ? 'k' : dv.toString();
  
  return verifier === dvCalculado;
}

/**
 * Formatea RUT chileno con puntos y guión
 * @param {string} rut - RUT sin formato
 * @returns {string} - RUT formateado
 */
function formatearRutChileno(rut) {
  const cleanRut = rut.replace(/[.-]/g, '');
  if (cleanRut.length < 8) return rut;
  
  const rutDigits = cleanRut.slice(0, -1);
  const verifier = cleanRut.slice(-1);
  
  // Formatear con puntos
  const formatted = rutDigits.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
  return `${formatted}-${verifier}`;
}

/**
 * Valida teléfono chileno (9 dígitos)
 * @param {string} telefono - Teléfono a validar
 * @returns {boolean} - true si es válido
 */
function validarTelefonoChileno(telefono) {
  if (!telefono) return false;
  
  // Remover espacios y caracteres especiales
  const cleanPhone = telefono.replace(/[\s-+()]/g, '');
  
  // Debe tener exactamente 9 dígitos
  if (cleanPhone.length !== 9) return false;
  
  // Debe ser solo números
  if (!/^\d{9}$/.test(cleanPhone)) return false;
  
  // Móviles chilenos empiezan con 9, 8, 7, 6 o 5
  // Fijos de Santiago empiezan con 2
  // Fijos de regiones empiezan con 3, 4, 5
  const firstDigit = cleanPhone[0];
  return ['2', '3', '4', '5', '6', '7', '8', '9'].includes(firstDigit);
}

/**
 * Formatea teléfono chileno
 * @param {string} telefono - Teléfono sin formato
 * @returns {string} - Teléfono formateado
 */
function formatearTelefonoChileno(telefono) {
  const cleanPhone = telefono.replace(/[\s-+()]/g, '');
  if (cleanPhone.length !== 9) return telefono;
  
  // Formato: XXX XXX XXX
  return cleanPhone.replace(/(\d{3})(\d{3})(\d{3})/, '$1 $2 $3');
}

/**
 * Validar formulario de cliente chileno
 * @param {Object} cliente - Datos del cliente
 * @returns {Object} - {valid: boolean, errors: string[]}
 */
function validarClienteChileno(cliente) {
  const errors = [];
  
  // Validar RUT
  if (!cliente.cedula) {
    errors.push('RUT es requerido');
  } else if (!validarRutChileno(cliente.cedula)) {
    errors.push('RUT no es válido (formato: XX.XXX.XXX-X)');
  }
  
  // Validar teléfono
  if (cliente.telefono && !validarTelefonoChileno(cliente.telefono)) {
    errors.push('Teléfono no es válido (9 dígitos)');
  }
  
  // Validar email
  if (cliente.email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(cliente.email)) {
      errors.push('Email no es válido');
    }
  }
  
  // Validar nombres y apellidos
  if (!cliente.nombres || cliente.nombres.trim().length < 2) {
    errors.push('Nombres debe tener al menos 2 caracteres');
  }
  
  if (!cliente.apellidos || cliente.apellidos.trim().length < 2) {
    errors.push('Apellidos debe tener al menos 2 caracteres');
  }
  
  return {
    valid: errors.length === 0,
    errors: errors
  };
}

// Regiones de Chile
const REGIONES_CHILE = [
  'Arica y Parinacota',
  'Tarapacá',
  'Antofagasta',
  'Atacama',
  'Coquimbo',
  'Valparaíso',
  'Metropolitana de Santiago',
  'O\'Higgins',
  'Maule',
  'Ñuble',
  'Biobío',
  'La Araucanía',
  'Los Ríos',
  'Los Lagos',
  'Aysén',
  'Magallanes y Antártica Chilena'
];

// Exportar funciones para uso en Node.js o navegador
if (typeof module !== 'undefined' && module.exports) {
  module.exports = {
    validarRutChileno,
    formatearRutChileno,
    validarTelefonoChileno,
    formatearTelefonoChileno,
    validarClienteChileno,
    REGIONES_CHILE
  };
} else {
  // Navegador
  window.ChileValidations = {
    validarRutChileno,
    formatearRutChileno,
    validarTelefonoChileno,
    formatearTelefonoChileno,
    validarClienteChileno,
    REGIONES_CHILE
  };
}
