import ApexCharts from "apexcharts";

// ===== chartTwo - Comparación anual de ventas
const chart02 = async () => {
  try {
    const idEmpresa = localStorage.getItem('id_empresa');
    const authToken = localStorage.getItem('authToken');
    
    if (!idEmpresa || !authToken) {
      console.warn('No se encontró ID de empresa o token para chart-02');
      renderFallbackChart();
      return;
    }

    // Usar el endpoint básico de ventas que sabemos que funciona
    const res = await fetch(`http://localhost:3001/api/ventas?empresa=${idEmpresa}`, {
      headers: {
        'Authorization': `Bearer ${authToken}`,
        'Content-Type': 'application/json'
      }
    });

    if (!res.ok) {
      throw new Error(`HTTP ${res.status}: ${res.statusText}`);
    }

    const ventasResponse = await res.json();
    
    console.log('🔍 Chart-02 - Respuesta del endpoint básico de ventas:', ventasResponse);

    // Verificar formato de respuesta
    let ventasBasicas;
    if (ventasResponse.success && Array.isArray(ventasResponse.data)) {
      ventasBasicas = ventasResponse.data;
    } else if (Array.isArray(ventasResponse)) {
      ventasBasicas = ventasResponse;
    } else {
      console.warn('Los datos de ventas no son válidos para chart-02:', ventasResponse);
      renderFallbackChart();
      return;
    }

    console.log('📊 Chart-02 - Datos de ventas procesados:', ventasBasicas);

    // Procesar datos para crear estructura de comparación anual
    const response = processVentasForComparison(ventasBasicas);
    
    console.log('📈 Chart-02 - Datos procesados para comparación:', response);

    if (!response.success) {
      console.warn('Los datos procesados no son válidos:', response);
      renderFallbackChart();
      return;
    }

    const {
      año_actual,
      año_anterior,
      mes_actual,
      datos_año_actual,
      datos_año_anterior,
      totales
    } = response;

    // Nombres de los meses
    const nombresMeses = [
      'Ene', 'Feb', 'Mar', 'Abr', 'May', 'Jun',
      'Jul', 'Ago', 'Sep', 'Oct', 'Nov', 'Dic'
    ];

    // Preparar datos para el gráfico (montos mensuales)
    const datosActuales = datos_año_actual.map(mes => mes.monto_total);
    const datosAnteriores = datos_año_anterior.map(mes => mes.monto_total);

    // Formatear montos en pesos chilenos
    const formatearPesos = (monto) => {
      return new Intl.NumberFormat('es-CL', {
        style: 'currency',
        currency: 'CLP',
        minimumFractionDigits: 0,
        maximumFractionDigits: 0
      }).format(monto);
    };

    // Actualizar elementos HTML con datos reales
    
    // Objetivo (ventas del año anterior completo)
    const objetivoElement = document.getElementById('objetivo-mes');
    if (objetivoElement) {
      objetivoElement.textContent = formatearPesos(totales.año_anterior);
    }

    // Ganancia actual (ventas del año actual completo)
    const gananciaElement = document.getElementById('ganancia-mes');
    if (gananciaElement) {
      gananciaElement.textContent = formatearPesos(totales.año_actual);
    }

    // Ganancia hasta el mes actual vs mismo período año anterior
    const gananciaHoyElement = document.getElementById('ganancia-hoy');
    if (gananciaHoyElement) {
      gananciaHoyElement.textContent = formatearPesos(totales.hasta_mes_actual);
    }

    // Actualizar porcentaje de cambio
    const avanceSpan = document.getElementById('avance-mes');
    if (avanceSpan) {
      const porcentajeCambio = Math.round(totales.porcentaje_cambio_mes);
      const signo = porcentajeCambio >= 0 ? '+' : '';
      avanceSpan.textContent = `${signo}${porcentajeCambio}%`;
      
      // Cambiar color según si es positivo o negativo
      avanceSpan.className = avanceSpan.className.replace(/bg-(success|red)-\d+|text-(success|red)-\d+/g, '');
      if (porcentajeCambio >= 0) {
        avanceSpan.className += ' bg-success-50 text-success-600 dark:bg-success-500/15 dark:text-success-500';
      } else {
        avanceSpan.className += ' bg-red-50 text-red-600 dark:bg-red-500/15 dark:text-red-500';
      }
    }

    // Actualizar mensaje
    const mensajeElement = document.getElementById('mensaje-ganancia');
    if (mensajeElement) {
      const diferencia = totales.diferencia_hasta_mes;
      const nombreMesActual = nombresMeses[mes_actual - 1];
      
      if (diferencia > 0) {
        mensajeElement.textContent = `Hasta ${nombreMesActual}, tienes ${formatearPesos(diferencia)} más que en ${año_anterior}. ¡Excelente progreso!`;
      } else if (diferencia < 0) {
        mensajeElement.textContent = `Hasta ${nombreMesActual}, tienes ${formatearPesos(Math.abs(diferencia))} menos que en ${año_anterior}. ¡Puedes mejorar!`;
      } else {
        mensajeElement.textContent = `Hasta ${nombreMesActual}, igual rendimiento que en ${año_anterior}. ¡Mantén el ritmo!`;
      }
    }

    // Calcular progreso para el gráfico radial (basado en el incremento/decremento real)
    const porcentajeCambioMes = Math.round(totales.porcentaje_cambio_mes);
    const progresoObjetivo = Math.abs(porcentajeCambioMes); // Usar el valor absoluto del cambio porcentual

    // Configurar gráfico radial
    const chartTwoOptions = {
      series: [progresoObjetivo],
      colors: [porcentajeCambioMes >= 0 ? "#039855" : "#D92D20"], // Verde para positivo, rojo para negativo
      chart: {
        fontFamily: "Outfit, sans-serif",
        type: "radialBar",
        height: 330,
        sparkline: {
          enabled: true,
        },
      },
      plotOptions: {
        radialBar: {
          startAngle: -90,
          endAngle: 90,
          hollow: {
            size: "80%",
          },
          track: {
            background: "#E4E7EC",
            strokeWidth: "100%",
            margin: 5,
          },
          dataLabels: {
            name: { show: false },
            value: {
              fontSize: "36px",
              fontWeight: "600",
              offsetY: 60,
              color: "#1D2939",
              formatter: function (val) {
                const signo = porcentajeCambioMes >= 0 ? '+' : '-';
                return `${signo}${Math.round(val)}%`;
              },
            },
          },
        },
      },
      fill: { type: "solid", colors: [porcentajeCambioMes >= 0 ? "#039855" : "#D92D20"] },
      stroke: { lineCap: "round" },
      labels: [`Cambio vs ${año_anterior}`],
    };

    const chartSelector = document.querySelectorAll("#chartTwo");
    if (chartSelector.length) {
      // Limpiar gráfico anterior si existe
      const existingChart = document.querySelector("#chartTwo");
      if (existingChart) {
        existingChart.innerHTML = '';
      }
      
      const chartTwo = new ApexCharts(
        document.querySelector("#chartTwo"),
        chartTwoOptions,
      );
      chartTwo.render();
    }

  } catch (error) {
    console.error('Error cargando datos de comparación anual:', error);
    renderFallbackChart();
  }
};

// Función para procesar datos básicos de ventas para comparación anual
const processVentasForComparison = (ventas) => {
  const añoActual = new Date().getFullYear();
  const añoAnterior = añoActual - 1;
  const mesActual = new Date().getMonth() + 1; // 1-12
  
  // Inicializar arrays para ambos años
  const datosAñoActual = Array.from({ length: 12 }, (_, index) => ({
    mes: index + 1,
    total_ventas: 0,
    monto_total: 0,
    precio_promedio: 0
  }));
  
  const datosAñoAnterior = Array.from({ length: 12 }, (_, index) => ({
    mes: index + 1,
    total_ventas: 0,
    monto_total: 0,
    precio_promedio: 0
  }));

  // Procesar ventas por año y mes
  const ventasPorAñoMes = {};
  
  ventas.forEach(venta => {
    if (!venta.fecha_venta) return;
    
    const fecha = new Date(venta.fecha_venta);
    const año = fecha.getFullYear();
    const mes = fecha.getMonth() + 1; // 1-12
    const precio = parseFloat(venta.precio_venta) || 0;
    
    if (año !== añoActual && año !== añoAnterior) return;
    
    const key = `${año}-${mes}`;
    if (!ventasPorAñoMes[key]) {
      ventasPorAñoMes[key] = {
        año,
        mes,
        ventas: [],
        total_ventas: 0,
        monto_total: 0
      };
    }
    
    ventasPorAñoMes[key].ventas.push(precio);
    ventasPorAñoMes[key].total_ventas++;
    ventasPorAñoMes[key].monto_total += precio;
  });
  
  // Llenar los datos procesados
  Object.values(ventasPorAñoMes).forEach(({ año, mes, total_ventas, monto_total, ventas }) => {
    const precio_promedio = ventas.length > 0 ? monto_total / ventas.length : 0;
    const mesIndex = mes - 1;
    
    const datosVenta = {
      mes,
      total_ventas,
      monto_total,
      precio_promedio
    };
    
    if (año === añoActual && mesIndex >= 0 && mesIndex < 12) {
      datosAñoActual[mesIndex] = datosVenta;
    } else if (año === añoAnterior && mesIndex >= 0 && mesIndex < 12) {
      datosAñoAnterior[mesIndex] = datosVenta;
    }
  });
  
  // Calcular totales anuales
  const totalActual = datosAñoActual.reduce((sum, mes) => sum + mes.monto_total, 0);
  const totalAnterior = datosAñoAnterior.reduce((sum, mes) => sum + mes.monto_total, 0);
  const ventasActual = datosAñoActual.reduce((sum, mes) => sum + mes.total_ventas, 0);
  const ventasAnterior = datosAñoAnterior.reduce((sum, mes) => sum + mes.total_ventas, 0);
  
  // Calcular totales hasta el mes actual
  const totalHastaMesActual = datosAñoActual.slice(0, mesActual).reduce((sum, mes) => sum + mes.monto_total, 0);
  const totalHastaMesAnterior = datosAñoAnterior.slice(0, mesActual).reduce((sum, mes) => sum + mes.monto_total, 0);
  
  return {
    success: true,
    año_actual: añoActual,
    año_anterior: añoAnterior,
    mes_actual: mesActual,
    datos_año_actual: datosAñoActual,
    datos_año_anterior: datosAñoAnterior,
    totales: {
      año_actual: totalActual,
      año_anterior: totalAnterior,
      ventas_año_actual: ventasActual,
      ventas_año_anterior: ventasAnterior,
      hasta_mes_actual: totalHastaMesActual,
      hasta_mes_anterior: totalHastaMesAnterior,
      diferencia_total: totalActual - totalAnterior,
      diferencia_hasta_mes: totalHastaMesActual - totalHastaMesAnterior,
      porcentaje_cambio: totalAnterior > 0 ? ((totalActual - totalAnterior) / totalAnterior) * 100 : 0,
      porcentaje_cambio_mes: totalHastaMesAnterior > 0 ? ((totalHastaMesActual - totalHastaMesAnterior) / totalHastaMesAnterior) * 100 : 0
    }
  };
};

// Función para renderizar gráfico de fallback
const renderFallbackChart = () => {
  const chartTwoOptions = {
    series: [75],
    colors: ["#cccccc"],
    chart: {
      fontFamily: "Outfit, sans-serif",
      type: "radialBar",
      height: 330,
      sparkline: {
        enabled: true,
      },
    },
    plotOptions: {
      radialBar: {
        startAngle: -90,
        endAngle: 90,
        hollow: {
          size: "80%",
        },
        track: {
          background: "#E4E7EC",
          strokeWidth: "100%",
          margin: 5,
        },
        dataLabels: {
          name: { show: false },
          value: {
            fontSize: "36px",
            fontWeight: "600",
            offsetY: 60,
            color: "#999999",
            formatter: function (val) {
              return "Sin datos";
            },
          },
        },
      },
    },
    fill: { type: "solid", colors: ["#cccccc"] },
    stroke: { lineCap: "round" },
    labels: ["Sin conexión"],
  };

  const chartSelector = document.querySelectorAll("#chartTwo");
  if (chartSelector.length) {
    const existingChart = document.querySelector("#chartTwo");
    if (existingChart) {
      existingChart.innerHTML = '';
    }
    
    const chartTwo = new ApexCharts(
      document.querySelector("#chartTwo"),
      chartTwoOptions,
    );
    chartTwo.render();
  }

  // Actualizar elementos con datos de fallback
  const elementos = [
    { id: 'objetivo-mes', text: 'Sin datos' },
    { id: 'ganancia-mes', text: 'Sin datos' },
    { id: 'ganancia-hoy', text: 'Sin datos' },
    { id: 'avance-mes', text: '0%' },
    { id: 'mensaje-ganancia', text: 'No se pudieron cargar los datos de comparación.' }
  ];

  elementos.forEach(({ id, text }) => {
    const element = document.getElementById(id);
    if (element) {
      element.textContent = text;
    }
  });
};

export default chart02;
