// Configuración global del sistema multi-empresa
window.EMPRESA_CONFIG = {
  // Obtener el ID de empresa del usuario actual
  getCurrentEmpresaId: () => {
    const id_empresa = localStorage.getItem('id_empresa');
    if (!id_empresa) {
      console.warn('No se encontró ID de empresa. Usuario no autenticado.');
      return null;
    }
    return parseInt(id_empresa);
  },
  
  // Verificar si el usuario pertenece a una empresa específica
  belongsToEmpresa: (empresaId) => {
    const currentEmpresa = window.EMPRESA_CONFIG.getCurrentEmpresaId();
    return currentEmpresa === parseInt(empresaId);
  },
  
  // Obtener información completa del usuario y empresa
  getUserAndEmpresa: () => {
    const userInfo = window.API_CONFIG.getUserInfo();
    if (!userInfo || !userInfo.id_empresa) return null;
    
    return {
      id_empresa: parseInt(userInfo.id_empresa),
      id_usuario: parseInt(userInfo.id_usuario),
      rol: userInfo.rol,
      usuario: userInfo.usuario,
      isAdmin: userInfo.rol === 'admin',
      isGerente: userInfo.rol === 'gerente',
      isTrabajador: userInfo.rol === 'trabajador'
    };
  },
  
  // Validar acceso según el rol
  canAccess: (requiredRole = 'trabajador') => {
    const user = window.EMPRESA_CONFIG.getUserAndEmpresa();
    if (!user) return false;
    
    const roleHierarchy = {
      'trabajador': 1,
      'gerente': 2,
      'admin': 3
    };
    
    const userLevel = roleHierarchy[user.rol] || 0;
    const requiredLevel = roleHierarchy[requiredRole] || 0;
    
    return userLevel >= requiredLevel;
  },
  
  // Obtener datos con filtro de empresa automático
  getEmpresaData: async (endpoint, additionalParams = {}) => {
    const empresaId = window.EMPRESA_CONFIG.getCurrentEmpresaId();
    if (!empresaId) {
      throw new Error('No se encontró ID de empresa para cargar datos');
    }
    
    const params = new URLSearchParams({
      empresa: empresaId,
      ...additionalParams
    });
    
    // Agregar /api al endpoint si no lo tiene
    const fullEndpoint = endpoint.startsWith('/api/') ? endpoint : `/api/${endpoint}`;
    
    // Usar el método request de API_CONFIG que incluye autenticación automática
    return await window.API_CONFIG.request(`${fullEndpoint}?${params}`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${window.API_CONFIG.getAuthToken()}`
      }
    });
  },
  
  // Enviar datos con empresa automática
  postEmpresaData: async (endpoint, data = {}) => {
    const empresaId = window.EMPRESA_CONFIG.getCurrentEmpresaId();
    if (!empresaId) {
      throw new Error('No se encontró ID de empresa para enviar datos');
    }
    
    const dataWithEmpresa = {
      ...data,
      id_empresa: empresaId
    };
    
    return await window.API_CONFIG.request(endpoint, {
      method: 'POST',
      body: JSON.stringify(dataWithEmpresa)
    });
  }
};

// Configuración de la aplicación
const config = {
  // URLs de la API - configurar según el entorno
  API_BASE_URL: window.location.hostname === 'localhost' 
    ? 'http://localhost:3001/api' 
    : 'https://tu-dominio-produccion.com/api',
  
  // Configuración de paginación
  ITEMS_PER_PAGE: 40,
  
  // Configuración de timeouts
  REQUEST_TIMEOUT: 10000, // 10 segundos
  
  // Configuración de sesión
  SESSION_TIMEOUT: 3600000, // 1 hora en milisegundos
  
  // Configuración de alertas
  ALERT_DURATION: 4000, // 4 segundos
  
  // Configuración de validación
  MIN_PASSWORD_LENGTH: 6,
  MAX_FILE_SIZE: 5 * 1024 * 1024, // 5MB
  ALLOWED_IMAGE_TYPES: ['image/jpeg', 'image/png', 'image/webp'],
  
  // Configuración de filtros
  DEFAULT_FILTERS: {
    search: '',
    region: '',
    edad: ''
  }
};

// Función para obtener la URL completa de la API
function getApiUrl(endpoint) {
  // Asegurarnos de que no hay doble /api/
  const path = endpoint.startsWith('/') ? endpoint : `/${endpoint}`;
  return `${config.API_BASE_URL}${path}`;
}

// Función para hacer peticiones HTTP con configuración común
async function apiRequest(endpoint, options = {}) {
  const url = getApiUrl(endpoint);
  const defaultOptions = {
    headers: {
      'Content-Type': 'application/json',
      ...options.headers
    },
    ...options
  };

  try {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), config.REQUEST_TIMEOUT);
    
    const response = await fetch(url, {
      ...defaultOptions,
      signal: controller.signal
    });
    
    clearTimeout(timeoutId);
    
    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }
    
    return await response.json();
  } catch (error) {
    if (error.name === 'AbortError') {
      throw new Error('Request timeout');
    }
    throw error;
  }
}

// Función para mostrar alertas
function mostrarAlerta(tipo, mensaje = '') {
  // Crear alerta usando SweetAlert2 si está disponible
  if (typeof Swal !== 'undefined') {
    const iconos = {
      'error': 'error',
      'warning': 'warning',
      'success': 'success',
      'info': 'info'
    };
    
    Swal.fire({
      icon: iconos[tipo] || 'info',
      title: tipo.charAt(0).toUpperCase() + tipo.slice(1),
      text: mensaje || `Mensaje de ${tipo}`,
      toast: true,
      position: 'top-end',
      showConfirmButton: false,
      timer: 3000,
      timerProgressBar: true
    });
  } else {
    // Fallback con console log
    console.log(`[${tipo.toUpperCase()}]: ${mensaje}`);
  }
}

// Función para validar sesión
function validarSesion() {
  console.log('🔍 Validando sesión...');
  
  const loginTime = localStorage.getItem('loginTime');  
  const authToken = localStorage.getItem('authToken');
  
  console.log('📋 Datos de sesión:', {
    loginTime,
    tokenPresente: authToken ? true : false,
    currentPath: window.location.pathname
  });
  
  if (!loginTime || !authToken) {
    console.log('❌ Sesión inválida: Faltan datos de login');
    redirectToLogin();
    return false;
  }
  
  // Convertir loginTime de ISO string a timestamp
  const loginTimestamp = new Date(loginTime).getTime();
  const currentTime = new Date().getTime();
  const elapsedTime = currentTime - loginTimestamp;
  
  console.log('⏰ Tiempos:', {
    loginTime,
    loginTimestamp,
    currentTime,
    elapsedTime,
    timeoutLimit: config.SESSION_TIMEOUT,
    hoursElapsed: (elapsedTime / (1000 * 60 * 60)).toFixed(2)
  });
  
  if (elapsedTime > config.SESSION_TIMEOUT) {
    console.log('❌ Sesión expirada');
    localStorage.clear();
    mostrarAlerta('warning', 'Sesión expirada. Redirigiendo al login...');
    setTimeout(() => redirectToLogin(), 2000);
    return false;
  }
  
  console.log('✅ Sesión válida');
  return true;
}

// Función para redirigir al login
function redirectToLogin() {
  window.location.href = './signin.html';
}

// Función para formatear fechas
function formatearFecha(fecha) {
  return new Date(fecha).toLocaleDateString('es-ES', {
    year: 'numeric',
    month: '2-digit',
    day: '2-digit'
  });
}

// Función para formatear moneda
function formatearMoneda(cantidad) {
  return new Intl.NumberFormat('es-CO', {
    style: 'currency',
    currency: 'COP',
    minimumFractionDigits: 0,
    maximumFractionDigits: 0
  }).format(cantidad);
}

// Función para validar archivos de imagen
function validarImagen(file) {
  if (!config.ALLOWED_IMAGE_TYPES.includes(file.type)) {
    throw new Error('Tipo de archivo no permitido. Solo se permiten imágenes JPEG, PNG y WebP.');
  }
  
  if (file.size > config.MAX_FILE_SIZE) {
    throw new Error(`El archivo es demasiado grande. Tamaño máximo: ${config.MAX_FILE_SIZE / 1024 / 1024}MB`);
  }
  
  return true;
}

// Función para debounce (útil para búsquedas)
function debounce(func, wait) {
  let timeout;
  return function executedFunction(...args) {
    const later = () => {
      clearTimeout(timeout);
      func(...args);
    };
    clearTimeout(timeout);
    timeout = setTimeout(later, wait);
  };
}

// Inicializar validación de sesión al cargar la página
document.addEventListener('DOMContentLoaded', function() {
  // Establecer valores por defecto para darkMode y sidebarToggle si no existen
  if (localStorage.getItem('darkMode') === null) {
    localStorage.setItem('darkMode', 'true');
    console.log('🌙 Estableciendo modo oscuro por defecto');
  }
  
  if (localStorage.getItem('sidebarToggle') === null) {
    localStorage.setItem('sidebarToggle', 'true');
    console.log('📱 Estableciendo sidebar colapsado por defecto');
  }
  
  // Solo validar sesión si no estamos en la página de login
  if (!window.location.pathname.includes('signin.html')) {
    // Pequeño retraso para evitar condición de carrera con el login
    setTimeout(() => {
      console.log('🚀 Iniciando validación automática de sesión');
      validarSesion();
    }, 100);
  }
});

// Exportar configuración para uso global
window.config = config;
window.apiRequest = apiRequest;
window.getApiUrl = getApiUrl;
window.mostrarAlerta = mostrarAlerta;
window.validarSesion = validarSesion;
window.formatearFecha = formatearFecha;
window.formatearMoneda = formatearMoneda;
window.validarImagen = validarImagen;
window.debounce = debounce;
