// Script para mostrar estado de conexión al servidor
(function() {
  let serverCheckInterval;
  let isServerOnline = false;
  
  // Función para verificar si el servidor está online
  async function checkServerStatus() {
    try {
      const response = await fetch(window.API_CONFIG?.BASE_URL + '/health' || 'http://localhost:3001/health', {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        timeout: 5000
      });
      
      if (response.ok) {
        const data = await response.json();
        if (data.success) {
          if (!isServerOnline) {
            showServerStatus('✅ Conectado al servidor', 'success');
            isServerOnline = true;
          }
          return true;
        }
      }
      throw new Error('Server response not ok');
    } catch (error) {
      if (isServerOnline) {
        showServerStatus('❌ Sin conexión al servidor', 'error');
        isServerOnline = false;
      }
      return false;
    }
  }
  
  // Función para mostrar el estado del servidor
  function showServerStatus(message, type) {
    // Remover notificación anterior si existe
    const existing = document.getElementById('server-status-notification');
    if (existing) existing.remove();
    
    // Crear nueva notificación
    const notification = document.createElement('div');
    notification.id = 'server-status-notification';
    notification.className = `fixed top-4 right-4 z-50 px-4 py-2 rounded-lg shadow-lg transition-all duration-300 ${
      type === 'success' 
        ? 'bg-green-500 text-white' 
        : 'bg-red-500 text-white'
    }`;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    // Auto-ocultar después de 3 segundos si es éxito
    if (type === 'success') {
      setTimeout(() => {
        if (notification.parentNode) {
          notification.style.opacity = '0';
          setTimeout(() => notification.remove(), 300);
        }
      }, 3000);
    }
  }
  
  // Iniciar verificación cuando el DOM esté listo
  document.addEventListener('DOMContentLoaded', function() {
    // Verificación inicial
    checkServerStatus();
    
    // Verificar cada 30 segundos
    serverCheckInterval = setInterval(checkServerStatus, 30000);
    
    // Mostrar instrucciones si el servidor no está disponible
    setTimeout(() => {
      if (!isServerOnline) {
        const helpMessage = document.createElement('div');
        helpMessage.id = 'server-help-message';
        helpMessage.className = 'fixed bottom-4 right-4 z-50 max-w-sm p-4 bg-yellow-100 border border-yellow-400 text-yellow-800 rounded-lg shadow-lg';
        helpMessage.innerHTML = `
          <div class="flex items-start">
            <div class="flex-shrink-0">
              <svg class="w-5 h-5 text-yellow-400" fill="currentColor" viewBox="0 0 20 20">
                <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"></path>
              </svg>
            </div>
            <div class="ml-3">
              <h3 class="text-sm font-medium">Servidor no disponible</h3>
              <div class="mt-2 text-xs">
                <p>Para solucionarlo:</p>
                <ol class="mt-1 list-decimal list-inside space-y-1">
                  <li>Ejecuta <code class="bg-yellow-200 px-1 rounded">iniciar-servidor.bat</code></li>
                  <li>O usa <code class="bg-yellow-200 px-1 rounded">npm run server</code></li>
                  <li>Verifica que XAMPP/MySQL esté ejecutándose</li>
                </ol>
              </div>
              <button onclick="this.parentElement.parentElement.parentElement.remove()" class="mt-2 text-xs underline hover:no-underline">
                Cerrar
              </button>
            </div>
          </div>
        `;
        document.body.appendChild(helpMessage);
      }
    }, 5000); // Mostrar después de 5 segundos si no hay conexión
  });
  
  // Limpiar interval cuando se cierre la página
  window.addEventListener('beforeunload', function() {
    if (serverCheckInterval) {
      clearInterval(serverCheckInterval);
    }
  });
})();
