/**
 * AGREGAR MOTO COMPLETA - JavaScript con Wizard Dinámico de Etapas
 * ====================================================================
 */

// Variables globales
let etapaActual = 1;
let totalEtapas = 3; // Se calculará dinámicamente
let datosUsuario = {};
let marcasDisponibles = [];
let modelosDisponibles = [];
let etapasDinamicas = [];
let datosCompletos = {}; // Para almacenar todos los datos del formulario
let documentosSubidos = []; // Para almacenar documentos antes del registro

// Configuración de etapas dinámicas
const ETAPAS_BASE = [
  { id: 1, nombre: 'Información Básica', requerida: true },
  { id: 2, nombre: 'Especificaciones Técnicas', requerida: true },
  { id: 3, nombre: 'Estado Comercial', requerida: true }
];

const ETAPAS_OPCIONALES = [
  { id: 'imagenes', nombre: 'Imágenes', requerida: false, icono: '📷' },
  { id: 'equipamiento', nombre: 'Equipamiento Adicional', requerida: false, icono: '⚙️' },
  { id: 'mantenciones', nombre: 'Mantenciones', requerida: false, icono: '🔧' },
  { id: 'confirmacion', nombre: 'Confirmación Final', requerida: true, icono: '✅' }
];

// ====================================================================
// VERIFICACIÓN DE AUTENTICACIÓN
// ====================================================================

function verificarAutenticacion() {
  const token = localStorage.getItem('authToken');
  
  if (!token) {
    console.error('❌ No se encontró token de autenticación');
    
    // Mostrar advertencia pero permitir continuar para testing
    Swal.fire({
      icon: 'warning',
      title: 'Sin autenticación',
      text: 'No se encontró token de autenticación. Las operaciones que requieran backend fallarán.',
      confirmButtonText: 'Continuar de todos modos',
      showCancelButton: true,
      cancelButtonText: 'Ir a Login'
    }).then((result) => {
      if (!result.isConfirmed) {
        window.location.href = '../signin.html';
      }
    });
    
    console.warn('⚠️ Continuando sin autenticación para desarrollo/testing');
    
    // Actualizar estado visual del botón
    setTimeout(() => {
      actualizarEstadoLogin();
    }, 100);
    
    return true; // Permitir continuar
  }
  
  // Verificar que el token no esté vacío
  if (token.trim() === '') {
    console.error('❌ Token de autenticación vacío');
    localStorage.removeItem('authToken'); // Limpiar token inválido
    console.warn('⚠️ Continuando sin autenticación para desarrollo/testing');
    
    // Actualizar estado visual del botón
    setTimeout(() => {
      actualizarEstadoLogin();
    }, 100);
    
    return true; // Permitir continuar
  }
  
  console.log('✅ Token de autenticación encontrado');
  
  // Actualizar estado visual del botón
  setTimeout(() => {
    actualizarEstadoLogin();
  }, 100);
  
  return true;
}

// ====================================================================
// INICIALIZACIÓN AUTOMÁTICA
// ====================================================================

document.addEventListener('DOMContentLoaded', async function() {
  // Solo ejecutar la inicialización automática si estamos en la página agregar-moto.html
  // En otros contextos (como consignaciones), se usará window.configurarFormularioAgregarMoto()
  if (!window.location.pathname.includes('consignaciones.html')) {
    console.log('🏍️ Iniciando módulo Agregar Moto Completo con Wizard Dinámico');
    
    // Verificar autenticación
    if (!verificarAutenticacion()) {
      return;
    }
    
    // Configurar tema inicial
    configurarTema();
    
    // Cargar datos del usuario
    cargarDatosUsuario();
    
    // Cargar datos externos
    await cargarMarcas();
    
    // Configurar fecha de ingreso por defecto
    configurarFechaDefecto();
    
    // Configurar eventos del formulario
    configurarEventosFormulario();
    
    // Inicializar wizard dinámico
    inicializarWizardDinamico();
    
    console.log('✅ Módulo Agregar Moto Completo inicializado correctamente');
  } else {
    console.log('🏍️ Agregar Moto cargado en contexto de integración (consignaciones)');
  }
});

function configurarEventosEspecificos() {
  // Eventos para cambio de marca
  const marcaSelect = document.getElementById('marcaMoto');
  if (marcaSelect) {
    marcaSelect.addEventListener('change', function(e) {
      const marcaId = e.target.value;
      if (marcaId) {
        cargarModelosPorMarca(marcaId);
      } else {
        limpiarSelectModelos();
      }
    });
  }
  
  // Eventos para validación de patente
  const patenteInput = document.getElementById('patenteMoto');
  if (patenteInput) {
    patenteInput.addEventListener('input', function(e) {
      const patente = e.target.value.toUpperCase();
      e.target.value = patente;
      
      if (patente.length >= 6) {
        validarPatenteEnTiempoReal(patente);
      }
    });
  }
  
  console.log('✅ Eventos específicos configurados');
}

async function validarPatenteLocal(patente) {
  // Usar función global validarPatenteEnTiempoReal
  return await validarPatenteEnTiempoReal(patente, 'patenteMoto', 'patente-feedback');
}

// Las funciones validarPatenteChile() y formatearPrecio(), formatearNumero(), formatearFecha() 
// ahora están disponibles desde global.js

function inicializarWizardDinamico() {
  // Inicializar etapas
  inicializarEtapas();
  
  // Mostrar primera etapa
  mostrarEtapa(1);
  
  // Actualizar progreso inicial
  actualizarBarraProgreso();
  
  console.log('🚀 Wizard dinámico inicializado');
}

function inicializarEtapas() {
  // Obtener etapas activas iniciales
  const etapasIniciales = obtenerEtapasActivas();
  etapasDinamicas = etapasIniciales;
  totalEtapas = etapasIniciales.length;
  
  console.log(`📋 Etapas inicializadas: ${totalEtapas} etapas activas`);
}

function mostrarEtapa(numeroEtapa) {
  // Ocultar todas las etapas
  document.querySelectorAll('.etapa').forEach(etapa => {
    etapa.classList.add('hidden');
  });
  
  // Mostrar la etapa solicitada
  const etapa = document.getElementById(`etapa${numeroEtapa}`);
  if (etapa) {
    etapa.classList.remove('hidden');
    etapaActual = numeroEtapa;
    
    // Actualizar progreso
    actualizarBarraProgreso();
    
    console.log(`📋 Mostrando etapa ${numeroEtapa}`);
  } else {
    console.error(`❌ No se encontró la etapa ${numeroEtapa}`);
  }
}

// Configurar tema inicial - Usar función global
// La función configurarTema() ahora está disponible desde global.js

// ====================================================================
// VERIFICACIÓN Y DATOS DEL USUARIO
// ====================================================================

// La función verificarAutenticacion() ahora está en global.js

function cargarDatosUsuario() {
  // Obtener datos del localStorage usando la clave correcta
  const datosGuardados = localStorage.getItem('datosUsuario');
  if (datosGuardados) {
    try {
      datosUsuario = JSON.parse(datosGuardados);
    } catch (error) {
      console.error('Error al parsear datos de usuario:', error);
      datosUsuario = {};
    }
  } else {
    console.warn('No se encontraron datos de usuario en localStorage');
    datosUsuario = {};
  }
  
  // Asegurar que tenemos id_empresa e id_usuario
  if (!datosUsuario.id_empresa) {
    datosUsuario.id_empresa = localStorage.getItem('id_empresa') || '1';
  }
  if (!datosUsuario.id_usuario) {
    datosUsuario.id_usuario = localStorage.getItem('id_usuario') || localStorage.getItem('userId') || '1';
  }
  
  console.log('👤 Datos usuario cargados:', datosUsuario);
  
  // Mostrar nombre de empresa en la etapa de confirmación
  const nombreEmpresa = datosUsuario.usuario?.empresa_nombre || 'Empresa no especificada';
  const elementoNombreEmpresa = document.getElementById('nombreEmpresa');
  if (elementoNombreEmpresa) {
    elementoNombreEmpresa.textContent = nombreEmpresa;
  }
}

// ====================================================================
// CONFIGURACIÓN DEL FORMULARIO
// ====================================================================

function configurarEventosFormulario() {
  const formulario = document.getElementById('formularioMotoCompleto');
  
  if (!formulario) {
    console.error('No se encontró el formulario con ID "formularioMotoCompleto"');
    return;
  }
  
  // Configurar evento para cambio de marca
  const marcaSelect = document.getElementById('marcaMoto');
  if (marcaSelect) {
    marcaSelect.addEventListener('change', function() {
      const marcaId = this.value;
      if (marcaId) {
        cargarModelosPorMarca(marcaId);
      } else {
        limpiarSelectModelos();
      }
    });
  }

  formulario.addEventListener('submit', async function(event) {
    event.preventDefault();
    
    // Solo procesar si estamos en la etapa de confirmación (999)
    if (etapaActual === 999) {
      // Mostrar confirmación antes de guardar
      const result = await Swal.fire({
        title: '¿Guardar moto?',
        text: 'Se registrará la motocicleta con todos los datos ingresados',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#10b981',
        cancelButtonColor: '#6b7280',
        confirmButtonText: '💾 Sí, guardar moto',
        cancelButtonText: 'Cancelar'
      });
      
      if (result.isConfirmed) {
        await agregarMotoCompleta();
      }
    } else {
      // Si no estamos en confirmación, avanzar a siguiente etapa
      siguienteEtapa();
    }
  });
  
  // Formatear patente automáticamente
  const patenteInput = document.getElementById('patenteMoto');
  if (patenteInput) {
    patenteInput.addEventListener('input', function(e) {
      e.target.value = e.target.value.toUpperCase();
    });
  } else {
    console.warn('No se encontró el elemento con ID "patenteMoto"');
  }
  
  // Formateo en tiempo real para precio
  const precioInput = document.getElementById('precioPublicacion');
  if (precioInput) {
    precioInput.addEventListener('input', function(e) {
      formatearPrecioChileno(e.target);
    });
  }
}

// ====================================================================
// NAVEGACIÓN ENTRE ETAPAS DINÁMICAS
// ====================================================================

function siguienteEtapa() {
  // Validar etapa actual antes de avanzar
  if (!validarEtapaActual()) {
    return;
  }
  
  // Determinar siguiente etapa
  const siguienteNumero = determinarSiguienteEtapa();
  if (!siguienteNumero) {
    console.log('No hay más etapas disponibles');
    return;
  }
  
  // Verificar si necesita generar etapas dinámicas
  verificarEtapasDinamicas();
  
  navegarAEtapa(siguienteNumero);
}

function anteriorEtapa() {
  const anteriorNumero = determinarAnteriorEtapa();
  if (!anteriorNumero) {
    console.log('No hay etapa anterior');
    return;
  }
  
  navegarAEtapa(anteriorNumero);
}

function navegarAEtapa(numeroEtapa) {
  // Animar salida de etapa actual
  const etapaActualElement = document.getElementById(`etapa${etapaActual}`);
  if (etapaActualElement) {
    etapaActualElement.classList.add('etapa-saliendo');
    
    setTimeout(() => {
      // Ocultar etapa actual
      etapaActualElement.classList.add('hidden');
      etapaActualElement.classList.remove('etapa-saliendo');
      
      // Mostrar nueva etapa
      const nuevaEtapa = document.getElementById(`etapa${numeroEtapa}`);
      if (nuevaEtapa) {
        nuevaEtapa.classList.remove('hidden');
        nuevaEtapa.classList.add('etapa-entrando');
        
        setTimeout(() => {
          nuevaEtapa.classList.remove('etapa-entrando');
        }, 50);
      } else {
        // Generar etapa si no existe
        generarEtapaDinamica(numeroEtapa);
      }
      
      // Actualizar estado
      etapaActual = numeroEtapa;
      actualizarBarraProgreso();
      
      // Ejecutar acciones específicas de la etapa
      ejecutarAccionesEtapa(numeroEtapa);
      
    }, 150);
  }
}

function ejecutarAccionesEtapa(numeroEtapa) {
  // Ejecutar acciones específicas según la etapa
  switch (numeroEtapa) {
    case 1:
      // Etapa de información básica
      configurarFechaDefecto();
      break;
    case 2:
      // Etapa de especificaciones
      break;
    case 3:
      // Etapa de estado comercial
      break;
    case 4:
      // Etapa de imágenes
      break;
    case 5:
      // Etapa de equipamiento
      break;
    case 6:
      // Etapa de mantenciones
      break;
    case 999:
      // Etapa de confirmación
      generarResumenDatos();
      break;
    default:
      console.log(`Etapa ${numeroEtapa} sin acciones específicas`);
  }
}

function determinarSiguienteEtapa() {
  const etapasActivas = obtenerEtapasActivas();
  const indiceActual = etapasActivas.findIndex(e => e.numero === etapaActual);
  
  if (indiceActual < etapasActivas.length - 1) {
    return etapasActivas[indiceActual + 1].numero;
  }
  
  return null; // No hay más etapas
}

function determinarAnteriorEtapa() {
  const etapasActivas = obtenerEtapasActivas();
  const indiceActual = etapasActivas.findIndex(e => e.numero === etapaActual);
  
  if (indiceActual > 0) {
    return etapasActivas[indiceActual - 1].numero;
  }
  
  return null; // No hay etapa anterior
}

function obtenerEtapasActivas() {
  let etapas = [
    { numero: 1, nombre: 'Información Básica', id: 'etapa1' },
    { numero: 2, nombre: 'Especificaciones Técnicas', id: 'etapa2' },
    { numero: 3, nombre: 'Estado Comercial', id: 'etapa3' },
    { numero: 4, nombre: 'Imágenes', id: 'etapa4' }
  ];

  // Etapas opcionales basadas en checkboxes
  if (document.getElementById('tieneEquipamiento')?.checked) {
    etapas.push({ numero: 5, nombre: 'Equipamiento', id: 'etapa5' });
  }

  if (document.getElementById('tieneMantenciones')?.checked) {
    etapas.push({ numero: 6, nombre: 'Mantenciones', id: 'etapa6' });
  }

  // Etapa final de confirmación
  etapas.push({ numero: 999, nombre: 'Confirmación', id: 'etapa999' });

  return etapas.sort((a, b) => a.numero - b.numero);
}

function verificarEtapasDinamicas() {
  // Verificar si hay cambios que requieren regenerar etapas
  const etapasActuales = obtenerEtapasActivas();
  
  if (JSON.stringify(etapasActuales) !== JSON.stringify(etapasDinamicas)) {
    etapasDinamicas = etapasActuales;
    totalEtapas = etapasActuales.length;
    generarIndicadorProgreso();
  }
}

// ====================================================================
// VALIDACIONES POR ETAPA
// ====================================================================

function validarEtapaActual() {
  switch (etapaActual) {
    case 1:
      return validarEtapa1();
    case 2:
      return validarEtapa2();
    case 3:
      return validarEtapa3();
    case 4:
      return validarEtapaImagenes();
    case 5:
      return validarEtapaEquipamiento();
    case 6:
      return validarEtapaMantenciones();
    case 999:
      return true; // Confirmación final
    default:
      return true;
  }
}

function validarEtapa1() {
  const campos = [
    { id: 'marcaMoto', nombre: 'Marca' },
    { id: 'modeloMoto', nombre: 'Modelo' },
    { id: 'añoMoto', nombre: 'Año' },
    { id: 'patenteMoto', nombre: 'Patente' },
    { id: 'condicionMoto', nombre: 'Condición' }
  ];
  
  for (const campo of campos) {
    const valor = document.getElementById(campo.id).value.trim();
    if (!valor) {
      mostrarError('Campo requerido', `El campo ${campo.nombre} es obligatorio.`);
      document.getElementById(campo.id).focus();
      return false;
    }
  }
  
  // Validar año
  const año = parseInt(document.getElementById('añoMoto').value);
  if (año < 1980 || año > 2030) {
    mostrarError('Año inválido', 'El año debe estar entre 1980 y 2030.');
    document.getElementById('añoMoto').focus();
    return false;
  }
  
  // Validar patente única
  return validarPatenteUnicaLocal();
}

function validarEtapa2() {
  const cilindrada = document.getElementById('cilindrada').value;
  if (!cilindrada || cilindrada < 50) {
    mostrarError('Cilindrada requerida', 'La cilindrada es obligatoria y debe ser mayor a 50cc.');
    document.getElementById('cilindrada').focus();
    return false;
  }
  
  return true;
}

function validarEtapa3() {
  const precio = obtenerValorNumericoPrecio('precioPublicacion');
  if (!precio || precio <= 0) {
    mostrarError('Precio requerido', 'El precio de publicación es obligatorio y debe ser mayor a 0.');
    document.getElementById('precioPublicacion').focus();
    return false;
  }
  
  return true;
}

function validarEtapaImagenes() {
  // Las imágenes son opcionales, siempre válida
  return true;
}

function validarEtapaEquipamiento() {
  // Validación básica de equipamientos agregados - excluir templates ocultos
  const equipamientos = document.querySelectorAll('.equipamiento-item:not(.hidden):not(#templateEquipamiento)');
  
  console.log('🔍 Validando equipamientos:', equipamientos.length, 'elementos encontrados');
  
  // Si no hay equipamientos agregados, la validación pasa
  if (equipamientos.length === 0) {
    console.log('✅ No hay equipamientos para validar');
    return true;
  }
  
  // Si hay equipamientos, validar que tengan descripción
  for (const equipamiento of equipamientos) {
    const descripcionElement = equipamiento.querySelector('.equipamiento-descripcion');
    if (!descripcionElement) {
      console.log('⚠️ Elemento sin campo de descripción encontrado');
      continue; // Saltar si el elemento no existe
    }
    
    const descripcion = descripcionElement.value;
    if (!descripcion.trim()) {
      mostrarError('Descripción requerida', `Debes completar la descripción del equipamiento. Valor actual: "${descripcion}"`);
      console.log('🔍 Validación equipamiento falló:', { elemento: equipamiento, descripcion: descripcion });
      return false;
    }
  }
  
  return true;
}

function validarEtapaMantenciones() {
  // Validación básica de mantenciones agregadas - excluir templates ocultos
  const mantenciones = document.querySelectorAll('.mantencion-item:not(.hidden):not(#templateMantencion)');
  console.log('🔍 Validando mantenciones - elementos encontrados:', mantenciones.length);
  
  // Si hay mantenciones, validar campos obligatorios
  for (const mantencion of mantenciones) {
    const fechaElement = mantencion.querySelector('.mantencion-fecha');
    const tipoElement = mantencion.querySelector('.mantencion-tipo');
    const descripcionElement = mantencion.querySelector('.mantencion-descripcion');
    const kilometrajeElement = mantencion.querySelector('.mantencion-kilometraje');
    const costoElement = mantencion.querySelector('.mantencion-costo');
    const realizadoPorElement = mantencion.querySelector('.mantencion-realizado-por');
    
    if (!fechaElement || !tipoElement || !descripcionElement || !kilometrajeElement || !costoElement || !realizadoPorElement) {
      continue; // Saltar si los elementos no existen
    }
    
    const fecha = fechaElement.value;
    const tipo = tipoElement.value;
    const descripcion = descripcionElement.value.trim();
    const kilometraje = kilometrajeElement.value;
    const costo = costoElement.value;
    const realizadoPor = realizadoPorElement.value.trim();
    
    // Validar campos obligatorios
    if (!fecha) {
      mostrarError('Fecha requerida', 'Debes especificar la fecha de la mantención.');
      fechaElement.focus();
      return false;
    }
    
    if (!tipo) {
      mostrarError('Tipo de mantención requerido', 'Debes seleccionar el tipo de mantención.');
      tipoElement.focus();
      return false;
    }
    
    if (!descripcion) {
      mostrarError('Descripción requerida', 'Debes proporcionar una descripción de la mantención.');
      descripcionElement.focus();
      return false;
    }
    
    if (!kilometraje || parseInt(kilometraje) < 0) {
      mostrarError('Kilometraje inválido', 'Debes especificar un kilometraje válido (mayor o igual a 0).');
      kilometrajeElement.focus();
      return false;
    }
    
    if (!costo || obtenerValorNumericoPrecio(costoElement) < 0) {
      mostrarError('Costo inválido', 'Debes especificar un costo válido (mayor o igual a 0).');
      costoElement.focus();
      return false;
    }
    
    if (!realizadoPor) {
      mostrarError('Ejecutor requerido', 'Debes especificar quién realizó la mantención.');
      realizadoPorElement.focus();
      return false;
    }
    
    // Validar que la fecha no sea futura
    const fechaMantencion = new Date(fecha);
    const hoy = new Date();
    hoy.setHours(23, 59, 59, 999); // Permitir fechas de hoy
    
    if (fechaMantencion > hoy) {
      mostrarError('Fecha inválida', 'La fecha de la mantención no puede ser futura.');
      fechaElement.focus();
      return false;
    }
  }
  
  return true;
}

async function validarPatenteUnicaLocal() {
  const patente = document.getElementById('patenteMoto').value.trim().toUpperCase();
  
  if (!patente) return true;
  
  // Usar función global
  return await validarPatenteUnica(patente, datosUsuario.id_empresa, apiRequest);
}

// ====================================================================
// BARRA DE PROGRESO DINÁMICA
// ====================================================================

function calcularEtapasDinamicas() {
  etapasDinamicas = [...ETAPAS_BASE];
  
  // Agregar etapas opcionales según configuración
  etapasDinamicas.push({ id: 4, nombre: 'Imágenes', requerida: false });
  
  // Etapas opcionales
  etapasDinamicas.push({ id: 5, nombre: 'Equipamiento', requerida: false });
  etapasDinamicas.push({ id: 6, nombre: 'Mantenciones', requerida: false });
  
  // Etapa final
  etapasDinamicas.push({ id: 999, nombre: 'Confirmación', requerida: true });
  
  totalEtapas = etapasDinamicas.length;
}

function generarIndicadorProgreso() {
  const contenedor = document.getElementById('contenedorPasos');
  if (!contenedor) return;
  
  const etapasActivas = obtenerEtapasActivas();
  let html = '';
  
  etapasActivas.forEach((etapa, index) => {
    const esUltimo = index === etapasActivas.length - 1;
    const estado = obtenerEstadoPaso(etapa.numero);
    
    html += `
      <div class="flex items-center ${index > 0 ? 'ml-4' : ''}">
        <div class="flex items-center">
          <div id="paso${etapa.numero}" class="w-8 h-8 md:w-10 md:h-10 rounded-full flex items-center justify-center text-sm font-bold transition-all duration-300 ${estado.clase}">
            ${estado.contenido}
          </div>
          <span class="ml-2 text-sm font-medium hidden md:inline ${estado.textoClase}">${etapa.nombre}</span>
        </div>
        ${!esUltimo ? `<div id="linea${etapa.numero}" class="flex-1 h-1 mx-2 md:mx-4 rounded transition-all duration-300 ${estado.lineaClase}"></div>` : ''}
      </div>
    `;
  });
  
  contenedor.innerHTML = html;
}

function obtenerEstadoPaso(numeroEtapa) {
  if (numeroEtapa < etapaActual) {
    return {
      clase: 'paso-completado',
      contenido: '✓',
      textoClase: 'text-green-600 dark:text-green-400',
      lineaClase: 'linea-completada'
    };
  } else if (numeroEtapa === etapaActual) {
    return {
      clase: 'paso-activo',
      contenido: numeroEtapa === 999 ? '✓' : numeroEtapa,
      textoClase: 'text-blue-600 dark:text-blue-400',
      lineaClase: 'linea-activa'
    };
  } else {
    return {
      clase: 'paso-pendiente',
      contenido: numeroEtapa === 999 ? '✓' : numeroEtapa,
      textoClase: 'text-gray-500 dark:text-gray-400',
      lineaClase: 'linea-pendiente'
    };
  }
}

function actualizarBarraProgreso() {
  const etapasActivas = obtenerEtapasActivas();
  const indiceActual = etapasActivas.findIndex(e => e.numero === etapaActual);
  const totalEtapasActivas = etapasActivas.length;
  
  // Actualizar texto de progreso
  const progresoTexto = document.getElementById('progresoTexto');
  if (progresoTexto) {
    progresoTexto.textContent = `Paso ${indiceActual + 1} de ${totalEtapasActivas}`;
  }
  
  // Actualizar barra de progreso porcentual
  const porcentaje = ((indiceActual + 1) / totalEtapasActivas) * 100;
  const barraProgreso = document.getElementById('barraProgresoPorcentaje');
  if (barraProgreso) {
    barraProgreso.style.width = `${porcentaje}%`;
  }
  
  // Actualizar indicadores de pasos dinámicos
  actualizarIndicadoresProgreso(etapasActivas, indiceActual);
}

function actualizarIndicadoresProgreso(etapasActivas, indiceActual) {
  const contenedorIndicadores = document.getElementById('contenedorPasos');
  if (!contenedorIndicadores) return;
  
  // Limpiar indicadores existentes
  contenedorIndicadores.innerHTML = '';
  
  // Generar indicadores dinámicos
  etapasActivas.forEach((etapa, index) => {
    const indicador = document.createElement('div');
    indicador.className = 'flex items-center';
    
    // Estado del paso
    let estadoClase = '';
    let iconoContent = '';
    
    if (index < indiceActual) {
      // Completado
      estadoClase = 'bg-green-500 text-white border-green-500';
      iconoContent = '✓';
    } else if (index === indiceActual) {
      // Actual
      estadoClase = 'bg-blue-500 text-white border-blue-500';
      iconoContent = index + 1;
    } else {
      // Pendiente
      estadoClase = 'bg-gray-300 text-gray-600 border-gray-300 dark:bg-gray-600 dark:text-gray-400 dark:border-gray-600';
      iconoContent = index + 1;
    }
    
    indicador.innerHTML = `
      <div class="flex items-center">
        <div class="w-8 h-8 rounded-full border-2 flex items-center justify-center text-sm font-semibold transition-all duration-300 ${estadoClase}">
          ${iconoContent}
        </div>
        ${index < etapasActivas.length - 1 ? `
          <div class="w-12 h-0.5 ${index < indiceActual ? 'bg-green-500' : 'bg-gray-300 dark:bg-gray-600'} transition-all duration-300"></div>
        ` : ''}
      </div>
    `;
    
    contenedorIndicadores.appendChild(indicador);
  });
}

// ====================================================================
// RESUMEN DE DATOS
// ====================================================================

function generarResumenDatos() {
  const datos = recopilarDatosFormulario();
  const contenidoResumen = document.getElementById('contenidoResumen');
  
  // Obtener información de marca, modelo y categoría desde los elementos del DOM
  const marcaElement = document.getElementById('marcaMoto');
  const modeloElement = document.getElementById('modeloMoto');
  const categoriaElement = document.getElementById('categoria');
  
  const marcaNombre = marcaElement?.selectedOptions[0]?.text || 'No especificada';
  const modeloNombre = modeloElement?.selectedOptions[0]?.text || 'No especificado';
  const categoriaNombre = categoriaElement?.selectedOptions[0]?.text || 'No especificada';
  
  // Debug: verificar datos recopilados
  console.log('🔍 Datos recopilados para resumen:', datos);
  console.log('🔍 Marca:', marcaNombre, 'Modelo:', modeloNombre, 'Categoría:', categoriaNombre);
  
  const html = `
    <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
      <div>
        <h5 class="font-semibold text-gray-900 dark:text-white mb-2">Información Básica</h5>
        <ul class="space-y-1 text-gray-700 dark:text-gray-300">
          <li><strong>Marca:</strong> ${marcaNombre}</li>
          <li><strong>Modelo:</strong> ${modeloNombre}</li>
          <li><strong>Categoría:</strong> ${categoriaNombre}</li>
          <li><strong>Año:</strong> ${datos.moto.año || 'No especificado'}</li>
          <li><strong>Color:</strong> ${datos.moto.color || 'No especificado'}</li>
          <li><strong>Kilometraje:</strong> ${formatearNumero(datos.moto.kms)} km</li>
          <li><strong>Patente:</strong> ${datos.moto.patente || 'No especificada'}</li>
          <li><strong>Llaves:</strong> ${datos.moto.llaves || 'No especificado'}</li>
          <li><strong>SOC:</strong> ${datos.moto.soc || 'No especificado'}</li>
        </ul>
      </div>
      <div>
        <h5 class="font-semibold text-gray-900 dark:text-white mb-2">Estado Comercial</h5>
        <ul class="space-y-1 text-gray-700 dark:text-gray-300">
          <li><strong>Precio Publicación:</strong> ${formatearPrecio(datos.estadoComercial.precio_publicacion)}</li>
          <li><strong>Estado:</strong> ${datos.estadoComercial.estado}</li>
          <li><strong>Condición:</strong> ${datos.estadoComercial.condicion}</li>
          <li><strong>Origen:</strong> ${datos.estadoComercial.origen}</li>
          <li><strong>Fecha Ingreso:</strong> ${formatearFecha(datos.estadoComercial.fecha_ingreso) || 'Hoy'}</li>
          ${datos.estadoComercial.precio_compra ? `<li><strong>Precio Compra:</strong> ${formatearPrecio(datos.estadoComercial.precio_compra)}</li>` : ''}
        </ul>
      </div>
    </div>
    ${datos.moto.observaciones ? `
      <div class="mt-4">
        <h5 class="font-semibold text-gray-900 dark:text-white mb-2">Observaciones</h5>
        <p class="text-gray-700 dark:text-gray-300 text-sm italic">"${datos.moto.observaciones}"</p>
      </div>
    ` : ''}
  `;
  
  contenidoResumen.innerHTML = html;
}

// ====================================================================
// RECOPILACIÓN DE DATOS COMPLETA
// ====================================================================

function recopilarDatosFormulario() {
  return {
    // Datos de la moto (tabla motos)
    moto: recopilarDatosMoto(),
    
    // Especificaciones técnicas (tabla especificaciones_tecnicas)
    especificaciones: recopilarEspecificaciones(),
    
    // Estado comercial (tabla estado_comercial)
    estadoComercial: recopilarEstadoComercial(),
    
    // Imágenes (tabla imagenes_motos)
    imagenes: recopilarImagenes(),
    
    // Equipamiento adicional (tabla equipamiento_adicional)
    equipamiento: recopilarEquipamiento(),
    
    // Mantenciones (tabla mantenciones)
    mantenciones: recopilarMantenciones(),
    
    // Datos de la empresa y usuario
    metadatos: {
      id_empresa: datosUsuario.id_empresa,
      id_usuario: datosUsuario.id_usuario,
      creado_por: datosUsuario.id_usuario
    }
  };
}

function recopilarDatosMoto() {
  return {
    id_empresa: datosUsuario.id_empresa,
    id_modelo: obtenerIdModelo(),
    año: parseInt(document.getElementById('añoMoto').value),
    color: document.getElementById('colorMoto').value.trim() || null,
    kms: parseInt(document.getElementById('kmsMoto').value) || 0,
    patente: document.getElementById('patenteMoto').value.trim().toUpperCase(),
    soc: document.getElementById('socMoto').value,
    llaves: parseInt(document.getElementById('llavesMoto').value) || 2,
    observaciones: document.getElementById('observacionesMoto').value.trim() || null
  };
}

function recopilarEspecificaciones() {
  return {
    cilindrada_cc: parseInt(document.getElementById('cilindrada').value),
    numero_motor: document.getElementById('numeroMotor').value.trim() || null,
    numero_chasis: document.getElementById('numeroChasis').value.trim() || null,
    tipo_motor: document.getElementById('tipoMotor').value || null
  };
}

function recopilarEstadoComercial() {
  return {
    precio_publicacion: obtenerValorNumericoPrecio('precioPublicacion'),
    estado: document.getElementById('estadoMoto')?.value || 'DISPONIBLE',
    condicion: document.getElementById('condicionMoto')?.value || 'NUEVA',
    origen: document.getElementById('origenMoto')?.value || 'COMPRA',
    fecha_ingreso: document.getElementById('fechaIngreso')?.value || new Date().toISOString().split('T')[0],
    creado_por: datosUsuario.id_usuario
  };
}

function recopilarImagenes() {
  const imagenes = [];
  const archivosImagenes = document.getElementById('imagenesFile')?.files;
  
  if (archivosImagenes && archivosImagenes.length > 0) {
    for (let i = 0; i < archivosImagenes.length; i++) {
      imagenes.push({
        archivo: archivosImagenes[i],
        es_principal: i === 0, // Primera imagen como principal
        tipo: 'MOTO'
      });
    }
  }
  
  return imagenes;
}

function recopilarEquipamiento() {
  const equipamientos = [];
  const items = document.querySelectorAll('.equipamiento-item:not(.hidden):not(#templateEquipamiento)');
  
  items.forEach(item => {
    const tipoElement = item.querySelector('.equipamiento-tipo');
    const descripcionElement = item.querySelector('.equipamiento-descripcion');
    
    if (tipoElement && descripcionElement) {
      const tipo = tipoElement.value;
      const descripcion = descripcionElement.value.trim();
      
      if (descripcion) {
        equipamientos.push({
          tipo_equipamiento: tipo,
          descripcion: descripcion,
          estado: 'BUENO', // Valor por defecto
          precio: 0, // Valor por defecto
          activo: true
        });
      }
    }
  });
  
  return equipamientos;
}

function recopilarMantenciones() {
  const mantenciones = [];
  const items = document.querySelectorAll('.mantencion-item:not(.hidden):not(#templateMantencion)');
  
  items.forEach(item => {
    const fechaElement = item.querySelector('.mantencion-fecha');
    const tipoElement = item.querySelector('.mantencion-tipo');
    const descripcionElement = item.querySelector('.mantencion-descripcion');
    const kilometrajeElement = item.querySelector('.mantencion-kilometraje');
    const costoElement = item.querySelector('.mantencion-costo');
    const realizadoPorElement = item.querySelector('.mantencion-realizado-por');
    const facturaElement = item.querySelector('.mantencion-factura');
    const imagenesElement = item.querySelector('.mantencion-imagenes');
    const tipoImagenElement = item.querySelector('.mantencion-tipo-imagen');
    
    if (fechaElement && tipoElement && descripcionElement) {
      const fecha = fechaElement.value;
      const tipo = tipoElement.value.trim();
      const descripcion = descripcionElement.value.trim();
      const kilometraje = kilometrajeElement ? parseInt(kilometrajeElement.value) || 0 : 0;
      const costo = costoElement ? obtenerValorNumericoPrecio(costoElement) || 0 : 0;
      const realizadoPor = realizadoPorElement ? realizadoPorElement.value.trim() || 'Taller interno' : 'Taller interno';
      const facturaNumero = facturaElement ? facturaElement.value.trim() || null : null;
      
      if (fecha && tipo && descripcion) {
        const mantencionData = {
          fecha: fecha,
          tipo_mantencion: tipo,
          descripcion: descripcion,
          kilometraje: kilometraje,
          costo: costo,
          realizado_por: realizadoPor,
          factura_numero: facturaNumero,
          creado_por: datosUsuario.id_usuario
        };
        
        // Agregar imágenes si existen
        if (imagenesElement && imagenesElement.files && imagenesElement.files.length > 0) {
          mantencionData.imagenes = [];
          const tipoImagen = tipoImagenElement ? tipoImagenElement.value : 'DESPUES';
          
          Array.from(imagenesElement.files).forEach(archivo => {
            mantencionData.imagenes.push({
              archivo: archivo,
              tipo_imagen: tipoImagen,
              nombre_archivo: archivo.name
            });
          });
        }
        
        mantenciones.push(mantencionData);
      }
    }
  });
  
  console.log('🔧 Mantenciones recopiladas:', mantenciones);
  return mantenciones;
}

// Función auxiliar para obtener valor numérico de precios con formato chileno
function obtenerValorNumericoPrecio(elementoOId) {
  let elemento;
  
  // Si es un string, obtener el elemento por ID
  if (typeof elementoOId === 'string') {
    elemento = document.getElementById(elementoOId);
  } else {
    // Si ya es un elemento, usarlo directamente
    elemento = elementoOId;
  }
  
  if (!elemento || !elemento.value) return 0;
  
  // Si es un input, usar el valor, si es otro elemento usar textContent
  const valor = elemento.value || elemento.textContent || '';
  
  // Convertir formato chileno (1.500.000) a número
  return parseInt(valor.replace(/\./g, '').replace(/[^\d]/g, '')) || 0;
}

function obtenerIdModelo() {
  const modeloSelect = document.getElementById('modeloMoto');
  
  if (!modeloSelect || !modeloSelect.value) {
    console.error('No se ha seleccionado un modelo válido');
    return null;
  }
  
  // Retornar el ID del modelo seleccionado
  const modeloId = parseInt(modeloSelect.value);
  console.log(`🏍️ Modelo seleccionado ID: ${modeloId}`);
  
  return modeloId;
}

// ====================================================================
// FUNCIONES DE CONFIRMACIÓN Y GUARDADO
// ====================================================================

async function confirmarGuardarMoto() {
  // Mostrar confirmación antes de guardar
  const result = await Swal.fire({
    title: '¿Guardar motocicleta?',
    text: 'Se registrará la motocicleta con todos los datos ingresados en el sistema',
    icon: 'question',
    showCancelButton: true,
    confirmButtonColor: '#10b981',
    cancelButtonColor: '#6b7280',
    confirmButtonText: '💾 Sí, guardar moto',
    cancelButtonText: 'Cancelar',
    showLoaderOnConfirm: true,
    preConfirm: async () => {
      try {
        await agregarMotoCompleta();
        return true;
      } catch (error) {
        Swal.showValidationMessage(`Error: ${error.message}`);
        return false;
      }
    },
    allowOutsideClick: () => !Swal.isLoading()
  });
}

async function confirmarLimpiarFormulario() {
  // Mostrar confirmación antes de limpiar
  const result = await Swal.fire({
    title: '¿Limpiar formulario?',
    text: 'Se perderán todos los datos ingresados. Esta acción no se puede deshacer.',
    icon: 'warning',
    showCancelButton: true,
    confirmButtonColor: '#f59e0b',
    cancelButtonColor: '#6b7280',
    confirmButtonText: '🗑️ Sí, limpiar todo',
    cancelButtonText: 'Cancelar'
  });
  
  if (result.isConfirmed) {
    limpiarFormulario();
  }
}

// ====================================================================
// PROCESAMIENTO COMPLETO DEL FORMULARIO
// ====================================================================

async function agregarMotoCompleta() {
  try {
    // Mostrar loading
    mostrarLoading('Registrando Moto', 'Procesando información completa...');

    const datosCompletos = recopilarDatosFormulario();
    
    console.log('📤 Enviando datos completos de moto:', datosCompletos);

    // 1. Crear/Obtener modelo si no existe
    const idModelo = await procesarModelo(datosCompletos);
    datosCompletos.moto.id_modelo = idModelo;

    // 2. Crear la moto principal
    const resultadoMoto = await apiRequest('/motos', {
      method: 'POST',
      body: JSON.stringify(datosCompletos.moto)
    });

    if (!resultadoMoto.success) {
      throw new Error(resultadoMoto.message || 'Error creando la moto');
    }

    const idMoto = resultadoMoto.id_moto;
    
    // 3. Crear especificaciones técnicas
    if (datosCompletos.especificaciones.cilindrada_cc) {
      datosCompletos.especificaciones.id_moto = idMoto;
      await apiRequest('/especificaciones', {
        method: 'POST',
        body: JSON.stringify(datosCompletos.especificaciones)
      });
    }

    // 4. Crear estado comercial
    datosCompletos.estadoComercial.id_moto = idMoto;
    await apiRequest('/estado-comercial', {
      method: 'POST',
      body: JSON.stringify(datosCompletos.estadoComercial)
    });

    // 5. Procesar imágenes si existen
    if (datosCompletos.imagenes.length > 0) {
      mostrarLoading('Subiendo Imágenes', `Procesando ${datosCompletos.imagenes.length} imagen(es)...`);
      await procesarImagenes(idMoto, datosCompletos.imagenes);
    }

    // 6. Procesar documentos si existen
    if (documentosSubidos.length > 0) {
      mostrarLoading('Subiendo Documentos', `Procesando ${documentosSubidos.length} documento(s)...`);
      await procesarDocumentos(idMoto);
    }

    // 7. Crear historial de dueños si aplica
    // 7. Crear equipamiento adicional si aplica
    if (datosCompletos.equipamiento.length > 0) {
      for (const equipo of datosCompletos.equipamiento) {
        equipo.id_moto = idMoto;
        await apiRequest('/equipamiento-adicional', {
          method: 'POST',
          body: JSON.stringify(equipo)
        });
      }
    }

    // 8. Crear mantenciones si aplica
    if (datosCompletos.mantenciones.length > 0) {
      mostrarLoading('Procesando Mantenciones', `Registrando ${datosCompletos.mantenciones.length} mantención(es)...`);
      
      for (const mantencion of datosCompletos.mantenciones) {
        // Separar las imágenes de los datos de la mantención
        const imagenesMantencion = mantencion.imagenes || [];
        const datosMantencion = { ...mantencion };
        delete datosMantencion.imagenes; // Remover imágenes de los datos principales
        
        // Crear la mantención primero
        const resultadoMantencion = await apiRequest(`/motos/${idMoto}/mantenciones`, {
          method: 'POST',
          body: JSON.stringify(datosMantencion)
        });
        
        // Si hay imágenes, subirlas después de crear la mantención
        if (imagenesMantencion.length > 0 && resultadoMantencion.success) {
          const idMantencion = resultadoMantencion.id_mantencion;
          await procesarImagenesMantenciones(idMantencion, imagenesMantencion);
        }
      }
    }

    // 10. Registrar en log de operaciones
    await registrarLogOperacion(idMoto, 'CREAR_MOTO', datosCompletos);

    cerrarLoading();
    
    await mostrarExito('¡Moto registrada exitosamente!', 
      `La moto ${datosCompletos.moto.patente} ha sido registrada completamente en el sistema.`);

    // Preguntar si desea agregar otra moto
    const resultado = await preguntarSiguienteAccion();
    
    if (resultado === 'otra') {
      limpiarFormularioCompleto();
    } else {
      volverAlDashboard();
    }

  } catch (error) {
    console.error('❌ Error registrando moto completa:', error);
    cerrarLoading();
    
    await mostrarError('Error en el registro', 
      error.message || 'Ocurrió un error durante el registro. Algunos datos pueden haberse guardado parcialmente.');
  }
}

async function procesarModelo(datos) {
  const marcaId = document.getElementById('marcaMoto').value;
  const modeloNombre = document.getElementById('modeloMoto').value.trim();
  const categoria = document.getElementById('categoria').value || 'CUSTOM';
  
  try {
    // Buscar modelo existente
    const modelosExistentes = await apiRequest(`/modelos?marca=${marcaId}&nombre=${modeloNombre}`);
    
    if (modelosExistentes && modelosExistentes.length > 0) {
      return modelosExistentes[0].id;
    }
    
    // Crear nuevo modelo
    const nuevoModelo = {
      id_marca: marcaId,
      nombre: modeloNombre,
      categoria: categoria,
      activo: true
    };
    
    const resultado = await apiRequest('/modelos', {
      method: 'POST',
      body: JSON.stringify(nuevoModelo)
    });
    
    return resultado.id;
    
  } catch (error) {
    console.error('Error procesando modelo:', error);
    return marcaId; // Fallback
  }
}

async function procesarImagenes(idMoto, imagenes) {
  if (!imagenes || imagenes.length === 0) {
    console.log('No hay imágenes para procesar');
    return;
  }

  const formData = new FormData();
  
  // Agregar todas las imágenes al FormData
  imagenes.forEach((imagen, index) => {
    formData.append('imagenes', imagen.archivo);
  });
  
  // Marcar la primera imagen como principal
  formData.append('es_principal', 'true');
  
  try {
    console.log(`📸 Subiendo ${imagenes.length} imagen(es) para moto ${idMoto}`);
    
    const response = await fetch(`/api/motos/${idMoto}/imagenes`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${localStorage.getItem('authToken')}`
      },
      body: formData
    });
    
    const resultado = await response.json();
    
    if (!response.ok) {
      throw new Error(resultado.message || 'Error subiendo imágenes');
    }
    
    console.log('✅ Imágenes subidas exitosamente:', resultado);
    return resultado;
    
  } catch (error) {
    console.error('❌ Error subiendo imágenes:', error);
    throw error;
  }
}

async function procesarImagenesMantenciones(idMantencion, imagenes) {
  if (!imagenes || imagenes.length === 0) {
    console.log('No hay imágenes para procesar en mantención');
    return;
  }

  const formData = new FormData();
  let tipoImagen = 'DESPUES'; // Valor por defecto
  
  // Agregar todas las imágenes al FormData
  imagenes.forEach((imagen, index) => {
    formData.append('imagenes', imagen.archivo);
    tipoImagen = imagen.tipo_imagen || 'DESPUES'; // Usar el tipo de la primera imagen
  });
  
  // Agregar el tipo de imagen
  formData.append('tipo_imagen', tipoImagen);
  
  try {
    console.log(`📸 Subiendo ${imagenes.length} imagen(es) para mantención ${idMantencion}`);
    
    const response = await fetch(`${window.API_CONFIG.BASE_URL}/api/mantenciones/${idMantencion}/imagenes`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${localStorage.getItem('authToken')}`
      },
      body: formData
    });
    
    const resultado = await response.json();
    
    if (!response.ok) {
      throw new Error(resultado.message || 'Error subiendo imágenes de mantención');
    }
    
    console.log('✅ Imágenes de mantención subidas exitosamente:', resultado);
    return resultado;
    
  } catch (error) {
    console.error('❌ Error subiendo imágenes de mantención:', error);
    // No lanzar error aquí porque las imágenes son opcionales
    console.warn('Las imágenes de mantención no se pudieron subir, pero la mantención se registró correctamente');
  }
}

async function registrarLogOperacion(idMoto, tipoOperacion, datos) {
  try {
    const logData = {
      id_usuario: datosUsuario.id_usuario,
      tipo_operacion: tipoOperacion,
      tabla_afectada: 'motos',
      id_registro: idMoto,
      detalle: JSON.stringify({
        patente: datos.moto.patente,
        marca: document.getElementById('marcaMoto').selectedOptions[0]?.text,
        modelo: datos.moto.modelo,
        año: datos.moto.año,
        precio: datos.estadoComercial.precio_publicacion,
        fecha_operacion: new Date().toISOString()
      })
    };
    
    await apiRequest('/logs', {
      method: 'POST',
      body: JSON.stringify(logData)
    });
    
    console.log('✅ Log de operación registrado exitosamente');
  } catch (error) {
    console.error('Error registrando log:', error);
    // No lanzar error aquí porque el log es opcional y no debe interrumpir el flujo principal
  }
}

async function preguntarSiguienteAccion() {
  const result = await Swal.fire({
    title: '¿Qué deseas hacer ahora?',
    text: 'La moto se registró correctamente',
    icon: 'success',
    showCancelButton: true,
    confirmButtonText: '🏍️ Agregar Otra Moto',
    cancelButtonText: '← Volver al Dashboard',
    confirmButtonColor: '#10b981',
    cancelButtonColor: '#6b7280'
  });
  
  if (result.isConfirmed) {
    return 'otra';
  } else {
    return 'dashboard';
  }
}

// ====================================================================
// CARGA DE DATOS EXTERNOS
// ====================================================================

async function cargarMarcas() {
  try {
    const marcas = await apiRequest('/marcas');
    if (Array.isArray(marcas)) {
      marcasDisponibles = marcas;
      llenarSelectMarcas();
    } else {
      console.error('Error al cargar marcas: formato de respuesta inválido');
      Swal.fire('Error', 'No se pudieron cargar las marcas', 'error');
    }
  } catch (error) {
    console.error('Error al cargar marcas:', error);
    Swal.fire('Error', 'Error de conexión al cargar marcas', 'error');
  }
}

function llenarSelectMarcas() {
  const select = document.getElementById('marcaMoto');
  if (!select) return;
  
  // Limpiar opciones existentes excepto la primera
  select.innerHTML = '<option value="">Seleccionar marca...</option>';
  
  // Agregar marcas
  marcasDisponibles.forEach(marca => {
    const option = document.createElement('option');
    option.value = marca.id;
    option.textContent = marca.nombre;
    select.appendChild(option);
  });
}

// Las funciones cargarTiposMantencion(), cargarTiposEquipamiento(), cargarTrabajadores(), 
// cargarRegiones(), validarPatenteExterna(), obtenerCotizacionDolar() y configurarFechaDefecto()
// ahora están disponibles desde global.js

// ====================================================================
// UTILIDADES Y FORMATEO
// ====================================================================

function limpiarFormulario() {
  // Resetear formulario con el ID correcto
  const formulario = document.getElementById('formularioMotoCompleto');
  if (formulario) {
    formulario.reset();
  }
  
  // Volver a la primera etapa
  document.querySelectorAll('.etapa').forEach(etapa => etapa.classList.add('hidden'));
  document.getElementById('etapa1').classList.remove('hidden');
  
  etapaActual = 1;
  actualizarBarraProgreso();
  
  // Configurar fecha por defecto nuevamente
  configurarFechaDefecto();
  
  // Mostrar confirmación de limpieza
  Swal.fire({
    title: '¡Formulario limpiado!',
    text: 'Todos los datos han sido eliminados',
    icon: 'success',
    timer: 2000,
    showConfirmButton: false
  });
  
  console.log('🧹 Formulario limpiado');
}

// Las funciones formatearPrecio(), formatearNumero(), formatearFecha() ahora están en global.js

function volverAlDashboard() {
  // Navegación relativa de vuelta al dashboard (puerto 3000)
  window.location.href = '../consignar-vender.html';
}

// ====================================================================
// EVENTOS ESPECÍFICOS Y INTERACCIONES
// ====================================================================

// Evento para cambio de marca
document.addEventListener('change', function(e) {
  if (e.target.id === 'marcaMoto') {
    const marcaId = e.target.value;
    if (marcaId) {
      cargarModelosPorMarca(marcaId);
    } else {
      limpiarSelectModelos();
    }
  }
});

// Evento para validación de patente en tiempo real
document.addEventListener('input', function(e) {
  if (e.target.id === 'patenteMoto') {
    const patente = e.target.value.trim();
    if (patente.length >= 3) {
      validarPatenteLocal(patente);
    }
  }
});

// Evento para formateo automático de precios
document.addEventListener('input', function(e) {
  if (e.target.id === 'precioPublicacion') {
    formatearPrecioChileno(e.target);
  }
});

// Evento para cálculo automático de kilometraje promedio
document.addEventListener('input', function(e) {
  if (['kmsMoto', 'añoMoto'].includes(e.target.id)) {
    calcularKilometrajePromedio();
  }
});

// Eventos para equipamientos
document.addEventListener('click', function(e) {
  if (e.target.id === 'btnAgregarEquipamiento') {
    agregarEquipamiento();
  } else if (e.target.classList.contains('btn-eliminar-equipamiento')) {
    eliminarEquipamiento(e.target);
  }
});

// Eventos para mantenciones
document.addEventListener('click', function(e) {
  if (e.target.id === 'btnAgregarMantencion') {
    agregarMantencion();
  } else if (e.target.classList.contains('btn-eliminar-mantencion')) {
    eliminarMantencion(e.target);
  } else if (e.target.classList.contains('btn-seleccionar-imagenes-mantencion')) {
    // El evento se maneja en configurarEventosMantencion
  }
});

// Eventos para formato de precios en mantenciones (delegación de eventos)
document.addEventListener('input', function(e) {
  if (e.target.classList.contains('mantencion-costo')) {
    formatearPrecioChileno(e.target);
  }
});

// Evento adicional para cambio de modelo (actualizar categoría automáticamente)
document.addEventListener('change', function(e) {
  if (e.target.id === 'modeloMoto') {
    const selectedOption = e.target.selectedOptions[0];
    if (selectedOption && selectedOption.getAttribute('data-categoria')) {
      const categoria = selectedOption.getAttribute('data-categoria');
      actualizarCategoriaAutomatica(categoria);
    } else {
      // Si no hay modelo seleccionado, limpiar categoría
      limpiarCategoria();
    }
  }
});

// ====================================================================
// FUNCIONES PARA MANEJO DE IMÁGENES
// ====================================================================

async function cargarImagenesExistentes(idMoto) {
  try {
    const response = await apiRequest(`/motos/${idMoto}/imagenes`);
    
    if (response.success && response.data.length > 0) {
      mostrarImagenesExistentes(response.data);
    }
  } catch (error) {
    console.error('Error cargando imágenes existentes:', error);
  }
}

function mostrarImagenesExistentes(imagenes) {
  const preview = document.getElementById('imagenesPreview');
  if (!preview) return;
  
  preview.innerHTML = '';
  
  imagenes.forEach((imagen, index) => {
    const div = document.createElement('div');
    div.className = 'relative group';
    div.innerHTML = `
      <img src="${imagen.url}" class="w-full h-24 object-cover rounded-lg border-2 border-gray-200 group-hover:border-blue-400 transition-colors">
      <button type="button" data-imagen-id="${imagen.id}" class="btn-eliminar-imagen absolute top-1 right-1 bg-red-500 text-white rounded-full w-6 h-6 flex items-center justify-center text-xs hover:bg-red-600 transition-colors opacity-0 group-hover:opacity-100">
        ×
      </button>
      ${imagen.es_principal ? '<div class="absolute bottom-1 left-1 bg-green-500 text-white text-xs px-2 py-1 rounded">Principal</div>' : ''}
      <div class="absolute bottom-1 right-1 bg-blue-500 text-white text-xs px-2 py-1 rounded">Guardada</div>
    `;
    
    preview.appendChild(div);
  });
  
  // Agregar eventos para eliminar imágenes existentes
  preview.querySelectorAll('.btn-eliminar-imagen').forEach(btn => {
    btn.addEventListener('click', () => eliminarImagenExistente(btn.dataset.imagenId, btn.closest('.relative')));
  });
}

async function eliminarImagenExistente(imagenId, elementoDiv) {
  try {
    const result = await Swal.fire({
      title: '¿Eliminar imagen?',
      text: 'Esta acción no se puede deshacer',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#ef4444',
      cancelButtonColor: '#6b7280',
      confirmButtonText: 'Sí, eliminar',
      cancelButtonText: 'Cancelar'
    });
    
    if (result.isConfirmed) {
      // Aquí necesitaríamos el ID de la moto, por ahora solo removemos visualmente
      elementoDiv.remove();
      
      Swal.fire({
        title: 'Imagen eliminada',
        text: 'La imagen ha sido eliminada exitosamente',
        icon: 'success',
        timer: 2000,
        showConfirmButton: false
      });
    }
  } catch (error) {
    console.error('Error eliminando imagen:', error);
    Swal.fire({
      title: 'Error',
      text: 'No se pudo eliminar la imagen',
      icon: 'error'
    });
  }
}

// Función para validar archivos de imagen
function validarArchivosImagen(files) {
  const errores = [];
  const tiposPermitidos = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
  const tamañoMaximo = 5 * 1024 * 1024; // 5MB
  
  Array.from(files).forEach((file, index) => {
    if (!tiposPermitidos.includes(file.type)) {
      errores.push(`Archivo ${index + 1}: Tipo no permitido (${file.type})`);
    }
    
    if (file.size > tamañoMaximo) {
      errores.push(`Archivo ${index + 1}: Tamaño excede 5MB (${(file.size / 1024 / 1024).toFixed(2)}MB)`);
    }
  });
  
  if (errores.length > 0) {
    Swal.fire({
      title: 'Archivos no válidos',
      html: errores.join('<br>'),
      icon: 'error'
    });
    return false;
  }
  
  return true;
}

// Eventos para imágenes
document.addEventListener('DOMContentLoaded', function() {
  const imagenesInput = document.getElementById('imagenesInput');
  const imagenesFile = document.getElementById('imagenesFile');
  
  if (imagenesInput && imagenesFile) {
    // Click en zona de drop
    imagenesInput.addEventListener('click', () => imagenesFile.click());
    
    // Drag and drop
    imagenesInput.addEventListener('dragover', (e) => {
      e.preventDefault();
      imagenesInput.classList.add('dragover');
    });
    
    imagenesInput.addEventListener('dragleave', () => {
      imagenesInput.classList.remove('dragover');
    });
    
    imagenesInput.addEventListener('drop', (e) => {
      e.preventDefault();
      imagenesInput.classList.remove('dragover');
      const files = e.dataTransfer.files;
      
      if (validarArchivosImagen(files)) {
        mostrarPreviewImagenes(files);
      }
    });
    
    // Selección de archivos
    imagenesFile.addEventListener('change', (e) => {
      if (validarArchivosImagen(e.target.files)) {
        mostrarPreviewImagenes(e.target.files);
      }
    });
  }

  // Eventos para documentos
  const documentosInput = document.getElementById('documentosInput');
  const documentosFile = document.getElementById('documentosFile');
  
  if (documentosInput && documentosFile) {
    // Click en zona de drop
    documentosInput.addEventListener('click', () => documentosFile.click());
    
    // Drag and drop
    documentosInput.addEventListener('dragover', (e) => {
      e.preventDefault();
      documentosInput.classList.add('dragover');
    });
    
    documentosInput.addEventListener('dragleave', () => {
      documentosInput.classList.remove('dragover');
    });
    
    documentosInput.addEventListener('drop', (e) => {
      e.preventDefault();
      documentosInput.classList.remove('dragover');
      const files = e.dataTransfer.files;
      
      if (files.length > 0) {
        documentosFile.files = files;
      }
    });
    
    // Selección de archivos
    documentosFile.addEventListener('change', (e) => {
      // El archivo se procesa cuando se hace clic en "Agregar Documento"
    });
  }
});

// ====================================================================
// FUNCIONES AUXILIARES EXISTENTES
// ====================================================================

async function cargarModelosPorMarca(marcaId) {
  const modeloSelect = document.getElementById('modeloMoto');
  
  if (!modeloSelect) {
    console.error('No se encontró el select de modelos');
    return;
  }

  if (!marcaId) {
    limpiarSelectModelos();
    return;
  }

  try {
    // Mostrar estado de carga
    modeloSelect.innerHTML = '<option value="">Cargando modelos...</option>';
    modeloSelect.disabled = true;

    // Llamar al endpoint de modelos
    const modelos = await apiRequest(`/modelos?marca=${marcaId}`);
    
    if (Array.isArray(modelos)) {
      modelosDisponibles = modelos;
      llenarSelectModelos();
    } else {
      console.error('Error al cargar modelos: formato de respuesta inválido');
      modeloSelect.innerHTML = '<option value="">Error al cargar modelos</option>';
      Swal.fire('Error', 'No se pudieron cargar los modelos para esta marca', 'error');
    }
  } catch (error) {
    console.error('Error al cargar modelos por marca:', error);
    modeloSelect.innerHTML = '<option value="">Error al cargar modelos</option>';
    Swal.fire('Error', 'Error de conexión al cargar modelos', 'error');
  }
}

function llenarSelectModelos() {
  const select = document.getElementById('modeloMoto');
  if (!select) return;
  
  // Limpiar opciones existentes
  select.innerHTML = '<option value="">Seleccionar modelo...</option>';
  
  // Habilitar el select
  select.disabled = false;
  
  // Agregar modelos
  modelosDisponibles.forEach(modelo => {
    const option = document.createElement('option');
    option.value = modelo.id;
    option.textContent = `${modelo.nombre} (${modelo.categoria})`;
    // Guardar la categoría como data attribute para uso posterior
    option.setAttribute('data-categoria', modelo.categoria);
    select.appendChild(option);
  });
  
  // Agregar evento para actualizar categoría cuando se seleccione un modelo
  select.addEventListener('change', function() {
    const selectedOption = this.selectedOptions[0];
    if (selectedOption && selectedOption.getAttribute('data-categoria')) {
      const categoria = selectedOption.getAttribute('data-categoria');
      actualizarCategoriaAutomatica(categoria);
    }
  });
  
  console.log(`✅ Cargados ${modelosDisponibles.length} modelos`);
}

function limpiarSelectModelos() {
  const modeloSelect = document.getElementById('modeloMoto');
  if (modeloSelect) {
    modeloSelect.value = '';
    modeloSelect.disabled = true;
    modeloSelect.innerHTML = '<option value="">Selecciona primero una marca</option>';
    
    // Limpiar array de modelos
    modelosDisponibles = [];
    
    // Limpiar también la categoría cuando se limpia el modelo
    limpiarCategoria();
  }
}

function actualizarCategoriaAutomatica(categoria) {
  const categoriaSelect = document.getElementById('categoria');
  
  if (categoriaSelect && categoria) {
    categoriaSelect.value = categoria;
    
    // Mostrar feedback visual de que se actualizó automáticamente
    categoriaSelect.style.backgroundColor = '#dcfce7'; // Verde claro
    categoriaSelect.style.borderColor = '#16a34a'; // Verde
    
    // Crear un pequeño indicador visual
    const indicador = document.createElement('span');
    indicador.textContent = '✓ Auto-completado';
    indicador.className = 'text-xs text-green-600 ml-2';
    indicador.id = 'categoria-auto-indicador';
    
    // Remover indicador anterior si existe
    const indicadorAnterior = document.getElementById('categoria-auto-indicador');
    if (indicadorAnterior) {
      indicadorAnterior.remove();
    }
    
    // Agregar nuevo indicador
    const labelCategoria = categoriaSelect.previousElementSibling;
    if (labelCategoria) {
      labelCategoria.appendChild(indicador);
    }
    
    // Remover estilos después de 3 segundos
    setTimeout(() => {
      categoriaSelect.style.backgroundColor = '';
      categoriaSelect.style.borderColor = '';
      if (document.getElementById('categoria-auto-indicador')) {
        document.getElementById('categoria-auto-indicador').remove();
      }
    }, 3000);
    
    console.log(`📂 Categoría actualizada automáticamente: ${categoria}`);
  }
}

function limpiarCategoria() {
  const categoriaSelect = document.getElementById('categoria');
  if (categoriaSelect) {
    categoriaSelect.value = '';
    categoriaSelect.style.backgroundColor = '';
    categoriaSelect.style.borderColor = '';
    
    // Remover indicador si existe
    const indicador = document.getElementById('categoria-auto-indicador');
    if (indicador) {
      indicador.remove();
    }
  }
}

function agregarEquipamiento() {
  const listaEquipamientos = document.getElementById('listaEquipamientos');
  const template = document.getElementById('templateEquipamiento');
  
  if (listaEquipamientos && template) {
    const nuevoEquipamiento = template.cloneNode(true);
    nuevoEquipamiento.id = '';
    nuevoEquipamiento.classList.remove('hidden');
    listaEquipamientos.appendChild(nuevoEquipamiento);
  }
}

function eliminarEquipamiento(boton) {
  const equipamiento = boton.closest('.equipamiento-item');
  if (equipamiento) {
    equipamiento.remove();
  }
}

function agregarMantencion() {
  const listaMantenciones = document.getElementById('listaMantenciones');
  const template = document.getElementById('templateMantencion');
  
  if (listaMantenciones && template) {
    const nuevaMantencion = template.cloneNode(true);
    nuevaMantencion.id = '';
    nuevaMantencion.classList.remove('hidden');
    
    // Configurar eventos para la nueva mantención
    configurarEventosMantencion(nuevaMantencion);
    
    listaMantenciones.appendChild(nuevaMantencion);
    
    console.log('✅ Nueva mantención agregada');
  }
}

function configurarEventosMantencion(mantencionElement) {
  // Configurar eventos para las imágenes
  const btnSeleccionarImagenes = mantencionElement.querySelector('.btn-seleccionar-imagenes-mantencion');
  const inputImagenes = mantencionElement.querySelector('.mantencion-imagenes');
  const previewContainer = mantencionElement.querySelector('.mantencion-imagenes-preview');
  
  if (btnSeleccionarImagenes && inputImagenes) {
    btnSeleccionarImagenes.addEventListener('click', () => {
      inputImagenes.click();
    });
    
    inputImagenes.addEventListener('change', (e) => {
      mostrarPreviewImagenesMantencion(e.target.files, previewContainer);
    });
  }
  
  // Configurar formato de precio para el campo costo
  const costoInput = mantencionElement.querySelector('.mantencion-costo');
  if (costoInput) {
    costoInput.addEventListener('input', function(e) {
      formatearPrecioChileno(e.target);
    });
  }
  
  // Configurar evento para eliminar
  const btnEliminar = mantencionElement.querySelector('.btn-eliminar-mantencion');
  if (btnEliminar) {
    btnEliminar.addEventListener('click', () => {
      eliminarMantencion(btnEliminar);
    });
  }
}

function mostrarPreviewImagenesMantencion(files, previewContainer) {
  if (!previewContainer || !files || files.length === 0) return;
  
  previewContainer.innerHTML = '';
  previewContainer.classList.remove('hidden');
  
  Array.from(files).forEach((file, index) => {
    if (file.type.startsWith('image/')) {
      const reader = new FileReader();
      reader.onload = function(e) {
        const div = document.createElement('div');
        div.className = 'relative group';
        div.innerHTML = `
          <img src="${e.target.result}" class="w-full h-20 object-cover rounded-lg border-2 border-gray-200 group-hover:border-blue-400 transition-colors">
          <button type="button" data-index="${index}" class="btn-eliminar-imagen-mantencion absolute top-1 right-1 bg-red-500 text-white rounded-full w-6 h-6 flex items-center justify-center text-xs hover:bg-red-600 transition-colors opacity-0 group-hover:opacity-100">
            ×
          </button>
          <div class="absolute bottom-1 left-1 bg-blue-500 text-white text-xs px-1 py-0.5 rounded text-center w-8">
            ${index + 1}
          </div>
        `;
        
        // Configurar evento para eliminar imagen individual
        const btnEliminarImg = div.querySelector('.btn-eliminar-imagen-mantencion');
        btnEliminarImg.addEventListener('click', () => {
          eliminarImagenMantencion(btnEliminarImg, previewContainer);
        });
        
        previewContainer.appendChild(div);
      };
      reader.readAsDataURL(file);
    }
  });
}

function eliminarImagenMantencion(boton, previewContainer) {
  const imagenDiv = boton.closest('.relative');
  if (imagenDiv) {
    imagenDiv.remove();
    
    // Si no quedan imágenes, ocultar el preview
    if (previewContainer.children.length === 0) {
      previewContainer.classList.add('hidden');
    }
  }
}

function eliminarMantencion(boton) {
  const mantencion = boton.closest('.mantencion-item');
  if (mantencion) {
    mantencion.remove();
  }
}

function mostrarPreviewImagenes(files) {
  const preview = document.getElementById('imagenesPreview');
  const fileInput = document.getElementById('imagenesFile');
  
  if (!preview) return;
  
  // Limpiar preview anterior
  preview.innerHTML = '';
  
  // Actualizar el input file con los archivos seleccionados
  if (fileInput && files) {
    fileInput.files = files;
  }
  
  Array.from(files).forEach((file, index) => {
    if (file.type.startsWith('image/')) {
      const reader = new FileReader();
      reader.onload = (e) => {
        const div = document.createElement('div');
        div.className = 'relative group';
        div.innerHTML = `
          <img src="${e.target.result}" class="w-full h-24 object-cover rounded-lg border-2 border-gray-200 group-hover:border-blue-400 transition-colors">
          <button type="button" class="absolute top-1 right-1 bg-red-500 text-white rounded-full w-6 h-6 flex items-center justify-center text-xs hover:bg-red-600 transition-colors opacity-0 group-hover:opacity-100">
            ×
          </button>
          ${index === 0 ? '<div class="absolute bottom-1 left-1 bg-green-500 text-white text-xs px-2 py-1 rounded">Principal</div>' : ''}
        `;
        
        // Evento para eliminar imagen
        div.querySelector('button').addEventListener('click', () => {
          div.remove();
          // Actualizar el input file removiendo este archivo
          actualizarInputFiles(file);
        });
        
        preview.appendChild(div);
      };
      reader.readAsDataURL(file);
    }
  });
}

function actualizarInputFiles(archivoAEliminar) {
  const fileInput = document.getElementById('imagenesFile');
  if (!fileInput || !fileInput.files) return;
  
  // Crear nuevo FileList sin el archivo eliminado
  const dt = new DataTransfer();
  Array.from(fileInput.files).forEach(file => {
    if (file !== archivoAEliminar) {
      dt.items.add(file);
    }
  });
  
  fileInput.files = dt.files;
  
  // Si no quedan archivos, limpiar preview
  if (fileInput.files.length === 0) {
    document.getElementById('imagenesPreview').innerHTML = '';
  }
}

function formatearPrecioEnTiempoReal(campo) {
  let valor = campo.value.replace(/[^\d]/g, '');
  if (valor) {
    valor = parseInt(valor).toLocaleString('es-CL');
    campo.value = valor;
  }
}

function formatearPrecioChileno(campo) {
  // Eliminar todo lo que no sea número
  let valor = campo.value.replace(/[^\d]/g, '');
  
  if (valor) {
    // Formatear con puntos como separadores de miles
    valor = parseInt(valor).toLocaleString('es-CL');
    campo.value = valor;
  }
}

function calcularKilometrajePromedio() {
  const kms = parseInt(document.getElementById('kmsMoto')?.value) || 0;
  const año = parseInt(document.getElementById('añoMoto')?.value) || new Date().getFullYear();
  const añoActual = new Date().getFullYear();
  const antigüedad = añoActual - año;
  
  if (antigüedad > 0 && kms > 0) {
    const promedio = Math.round(kms / antigüedad);
    console.log(`Kilometraje promedio: ${promedio.toLocaleString()} km/año`);
  }
}

// ====================================================================
// LIMPIEZA DE FORMULARIO
// ====================================================================

function limpiarFormularioCompleto() {
  // Limpiar todos los campos del formulario con el ID correcto
  const formulario = document.getElementById('formularioMotoCompleto');
  if (formulario) {
    formulario.reset();
    console.log('✅ Formulario reseteado correctamente');
  } else {
    console.warn('⚠️ No se encontró el formulario para resetear');
  }
  
  // Resetear variables de estado
  etapaActual = 1;
  datosCompletos = {};
  
  // Limpiar arrays de datos si existen
  if (typeof marcasDisponibles !== 'undefined') marcasDisponibles = [];
  if (typeof modelosDisponibles !== 'undefined') modelosDisponibles = [];
  
  // Limpiar selects de modelo
  limpiarSelectModelos();
  
  // Volver a primera etapa
  mostrarEtapa(1);
  
  // Recalcular etapas dinámicas
  inicializarEtapas();
  
  // Reconfigurar fecha por defecto
  if (typeof configurarFechaDefecto === 'function') {
    configurarFechaDefecto();
  }
  
  // Recargar datos dinámicos
  if (typeof cargarMarcas === 'function') {
    cargarMarcas();
  }
  
  console.log('🧹 Formulario completo limpiado');
}

// ====================================================================
// GENERACIÓN DINÁMICA DE ETAPAS
// ====================================================================

function generarEtapaDinamica(numeroEtapa, tipoEtapa) {
  const contenedor = document.getElementById(`etapa${numeroEtapa}`);
  if (!contenedor) return;

  let contenidoHTML = '';

  switch (tipoEtapa) {
    case 'especificaciones':
      contenidoHTML = generarEtapaEspecificaciones();
      break;
    case 'historial':
      contenidoHTML = generarEtapaHistorial();
      break;
    case 'imagenes':
      contenidoHTML = generarEtapaImagenes();
      break;
    case 'equipamiento':
      contenidoHTML = generarEtapaEquipamiento();
      break;
    case 'mantenciones':
      contenidoHTML = generarEtapaMantenciones();
      break;
    case 'estado-comercial':
      contenidoHTML = generarEtapaEstadoComercial();
      break;
    case 'confirmacion':
      contenidoHTML = generarEtapaConfirmacion();
      break;
    default:
      contenidoHTML = '<div class="text-center py-8"><p class="text-gray-500">Etapa no definida</p></div>';
  }

  contenedor.innerHTML = contenidoHTML;
}

function generarEtapaEspecificaciones() {
  return `
    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
      <div>
        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          Cilindrada (cc)
        </label>
        <input type="number" id="cilindrada_cc" 
               class="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 dark:text-white">
      </div>
      
      <div>
        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          Potencia (HP)
        </label>
        <input type="number" id="potencia_hp" 
               class="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 dark:text-white">
      </div>
      
      <div>
        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          Peso (kg)
        </label>
        <input type="number" id="peso_kg" 
               class="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 dark:text-white">
      </div>
      
      <div>
        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          Capacidad Combustible (L)
        </label>
        <input type="number" id="capacidad_combustible_l" 
               class="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 dark:text-white">
      </div>
      
      <div class="md:col-span-2">
        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          Observaciones Técnicas
        </label>
        <textarea id="observaciones_tecnicas" 
                  class="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 dark:text-white" 
                  rows="3"></textarea>
      </div>
    </div>
  `;
}

function generarEtapaImagenes() {
  return `
    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
      <div>
        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          Subir Imágenes
        </label>
        <div class="dropzone" id="imagenesInput">
          Arrastra y suelta tus imágenes aquí o haz clic para seleccionarlas.
        </div>
      </div>
    </div>
  `;
}

function generarEtapaConfirmacion() {
  const datos = recopilarDatosFormulario();

  return `
    <div class="space-y-6">
      <div class="text-center">
        <h3 class="text-2xl font-bold text-gray-900 dark:text-white mb-2">
          🔍 Confirmación de Datos
        </h3>
        <p class="text-gray-600 dark:text-gray-400">
          Revisa toda la información antes de registrar la moto
        </p>
      </div>

      <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div class="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
          <h4 class="font-semibold text-gray-900 dark:text-white mb-2">Información Básica</h4>
          <p class="text-gray-700 dark:text-gray-300"><strong>Patente:</strong> ${datos.moto.patente || 'No especificada'}</p>
          <p class="text-gray-700 dark:text-gray-300"><strong>Marca:</strong> ${document.getElementById('marcaMoto')?.selectedOptions[0]?.text || 'No especificada'}</p>
          <p class="text-gray-700 dark:text-gray-300"><strong>Modelo:</strong> ${datos.moto.modelo || 'No especificado'}</p>
          <p class="text-gray-700 dark:text-gray-300"><strong>Año:</strong> ${datos.moto.año || 'No especificado'}</p>
          <p class="text-gray-700 dark:text-gray-300"><strong>Kilometraje:</strong> ${formatearNumero(datos.moto.kilometraje) || '0'} km</p>
        </div>

        <div class="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
          <h4 class="font-semibold text-gray-900 dark:text-white mb-2">Estado Comercial</h4>
          <p class="text-gray-700 dark:text-gray-300"><strong>Precio:</strong> ${formatearPrecio(datos.estadoComercial.precio_publicacion)}</p>
          <p class="text-gray-700 dark:text-gray-300"><strong>Estado:</strong> ${datos.estadoComercial.estado_moto || 'No especificado'}</p>
          <p class="text-gray-700 dark:text-gray-300"><strong>Tipo:</strong> ${datos.estadoComercial.tipo_operacion || 'No especificado'}</p>
        </div>
      </div>

      <div class="text-center">
        <button onclick="agregarMotoCompleta()" 
                class="bg-green-500 hover:bg-green-600 text-white font-bold py-3 px-8 rounded-lg">
          Confirmar Registro
        </button>
      </div>
    </div>
  `;
}

// ====================================================================
// GESTIÓN DE DOCUMENTOS
// ====================================================================

/**
 * Cambia entre tabs de imágenes y documentos
 */
function cambiarTab(tab) {
  // Remover clase active de todos los tabs
  document.querySelectorAll('.tab-button').forEach(btn => {
    btn.classList.remove('active');
    btn.classList.add('border-transparent', 'text-gray-500');
    btn.classList.remove('border-blue-500', 'text-blue-600');
  });
  
  // Ocultar todos los contenidos
  document.querySelectorAll('.tab-content').forEach(content => {
    content.classList.add('hidden');
  });
  
  // Activar tab seleccionado
  const tabButton = document.getElementById(`tab${tab.charAt(0).toUpperCase() + tab.slice(1)}`);
  const tabContent = document.getElementById(`contenido${tab.charAt(0).toUpperCase() + tab.slice(1)}`);
  
  if (tabButton && tabContent) {
    tabButton.classList.add('active');
    tabButton.classList.remove('border-transparent', 'text-gray-500');
    tabButton.classList.add('border-blue-500', 'text-blue-600');
    
    tabContent.classList.remove('hidden');
  }
}

/**
 * Agrega un documento a la lista temporal
 */
function agregarDocumento() {
  const tipoDocumento = document.getElementById('tipoDocumento').value;
  const descripcionDocumento = document.getElementById('descripcionDocumento').value;
  const archivoInput = document.getElementById('documentosFile');
  
  // Validaciones
  if (!tipoDocumento) {
    mostrarError('Error', 'Debe seleccionar un tipo de documento');
    return;
  }
  
  if (!archivoInput.files || archivoInput.files.length === 0) {
    mostrarError('Error', 'Debe seleccionar un archivo');
    return;
  }
  
  const archivo = archivoInput.files[0];
  
  // Validar tamaño (máximo 10MB)
  if (archivo.size > 10 * 1024 * 1024) {
    mostrarError('Error', 'El archivo no puede ser mayor a 10MB');
    return;
  }
  
  // Validar tipo de archivo
  const tiposPermitidos = [
    'application/pdf',
    'application/msword',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'image/jpeg',
    'image/jpg',
    'image/png'
  ];
  
  if (!tiposPermitidos.includes(archivo.type)) {
    mostrarError('Error', 'Tipo de archivo no permitido. Use PDF, DOC, DOCX, JPG o PNG');
    return;
  }
  
  // Verificar si ya existe un archivo con el mismo nombre
  const existeArchivo = documentosSubidos.some(doc => doc.nombre_archivo === archivo.name);
  if (existeArchivo) {
    mostrarError('Error', 'Ya existe un documento con el mismo nombre. Por favor renombre el archivo o seleccione otro.');
    return;
  }
  
  // Crear objeto documento
  const documento = {
    id: Date.now() + Math.random(), // ID temporal más único
    tipo_documento: tipoDocumento,
    descripcion: descripcionDocumento || `Documento ${tipoDocumento}`,
    archivo: archivo,
    nombre_archivo: archivo.name,
    tamaño_archivo: archivo.size,
    mime_type: archivo.type
  };
  
  // Agregar a la lista temporal
  documentosSubidos.push(documento);
  
  // Actualizar la vista
  actualizarListaDocumentos();
  
  // Limpiar formulario
  document.getElementById('tipoDocumento').value = '';
  document.getElementById('descripcionDocumento').value = '';
  document.getElementById('documentosFile').value = '';
  
  console.log('✅ Documento agregado:', documento.nombre_archivo);
}

/**
 * Actualiza la lista visual de documentos
 */
function actualizarListaDocumentos() {
  const listaContainer = document.getElementById('listaDocumentos');
  
  if (documentosSubidos.length === 0) {
    listaContainer.innerHTML = `
      <div class="text-center text-gray-500 dark:text-gray-400 py-8">
        <svg class="mx-auto h-12 w-12 text-gray-300 dark:text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
        </svg>
        <p class="text-sm mt-2">No hay documentos agregados</p>
      </div>
    `;
    return;
  }
  
  const documentosHTML = documentosSubidos.map(doc => {
    const iconoTipo = obtenerIconoTipoDocumento(doc.tipo_documento);
    const tamaño = formatearTamaño(doc.tamaño_archivo);
    
    return `
      <div class="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600">
        <div class="flex items-center space-x-3">
          <div class="text-2xl">${iconoTipo}</div>
          <div>
            <p class="text-sm font-medium text-gray-900 dark:text-white">${doc.nombre_archivo}</p>
            <p class="text-xs text-gray-500 dark:text-gray-400">${doc.descripcion} • ${tamaño}</p>
          </div>
        </div>
        <button onclick="eliminarDocumento(${doc.id})" 
                class="text-red-500 hover:text-red-700 p-1 rounded">
          <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
          </svg>
        </button>
      </div>
    `;
  }).join('');
  
  listaContainer.innerHTML = documentosHTML;
}

/**
 * Elimina un documento de la lista temporal
 */
function eliminarDocumento(id) {
  documentosSubidos = documentosSubidos.filter(doc => doc.id !== id);
  actualizarListaDocumentos();
  console.log('🗑️ Documento eliminado');
}

/**
 * Obtiene el icono apropiado para cada tipo de documento
 */
function obtenerIconoTipoDocumento(tipo) {
  const iconos = {
    'NOTA_VENTA': '📝',
    'CONSIGNACION': '📋',
    'FACTURA': '🧾',
    'BOLETA': '🧾',
    'CONTRATO': '📑',
    'GARANTIA': '🛡️',
    'SEGURO': '🚗',
    'REVISION_TECNICA': '🔍',
    'PERMISO_CIRCULACION': '🎫',
    'OTRO': '📄'
  };
  return iconos[tipo] || '📄';
}

/**
 * Formatea el tamaño de archivo
 */
function formatearTamaño(bytes) {
  if (bytes === 0) return '0 Bytes';
  const k = 1024;
  const sizes = ['Bytes', 'KB', 'MB', 'GB'];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

/**
 * Procesa los documentos durante el registro de la moto
 */
async function procesarDocumentos(motoId) {
  if (documentosSubidos.length === 0) {
    console.log('📄 No hay documentos para procesar');
    return;
  }
  
  // Verificar si hay token antes de procesar
  const token = localStorage.getItem('authToken');
  if (!token) {
    console.warn('⚠️ No hay token de autenticación, saltando subida de documentos');
    mostrarError('Sin Autenticación', 'No se pueden subir documentos sin un token de autenticación válido. Ve a la página de login para autenticarte.');
    return;
  }
  
  console.log(`📄 Procesando ${documentosSubidos.length} documentos para moto ${motoId}`);
  
  let documentosExitosos = 0;
  let documentosFallidos = 0;
  
  for (const documento of documentosSubidos) {
    try {
      const formData = new FormData();
      formData.append('archivo', documento.archivo);
      formData.append('tipo_documento', documento.tipo_documento);
      formData.append('descripcion', documento.descripcion);
      formData.append('tipo_entidad', 'MOTO');
      formData.append('id_entidad', motoId);
      
      const response = await fetch('/api/documentos', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`
        },
        body: formData
      });
      
      if (!response.ok) {
        if (response.status === 403) {
          throw new Error(`Token de autenticación inválido o expirado`);
        } else if (response.status === 401) {
          throw new Error(`No autorizado - verifica tu sesión`);
        } else {
          throw new Error(`Error ${response.status}: ${response.statusText}`);
        }
      }
      
      const resultado = await response.json();
      console.log(`✅ Documento subido: ${documento.nombre_archivo}`, resultado);
      documentosExitosos++;
      
    } catch (error) {
      console.error(`❌ Error subiendo documento ${documento.nombre_archivo}:`, error);
      documentosFallidos++;
      
      // Si es error de autenticación, mostrar mensaje específico
      if (error.message.includes('Token') || error.message.includes('autorizado')) {
        mostrarError('Error de Autenticación', 
          'Tu sesión ha expirado o no tienes permisos. Por favor, inicia sesión nuevamente.');
        break; // No intentar subir más documentos
      }
    }
  }
  
  // Mostrar resumen final
  if (documentosExitosos > 0) {
    console.log(`✅ ${documentosExitosos} documentos subidos exitosamente`);
  }
  if (documentosFallidos > 0) {
    console.warn(`⚠️ ${documentosFallidos} documentos fallaron`);
  }
  
  // Limpiar la lista después del procesamiento
  documentosSubidos = [];
  actualizarListaDocumentos();
}

// ====================================================================
// INICIALIZACIÓN DE LA PÁGINA
// ====================================================================

// Inicializar tabs por defecto cuando se carga la página
document.addEventListener('DOMContentLoaded', function() {
  // Configurar tab de imágenes como activo por defecto
  setTimeout(() => {
    cambiarTab('imagenes');
  }, 100);
});

// ====================================================================
// FUNCIONES DE AUTENTICACIÓN Y LOGIN
// ====================================================================

/**
 * Actualiza el estado visual del botón de login
 */
function actualizarEstadoLogin() {
  const boton = document.getElementById('botonLogin');
  const textoBoton = document.getElementById('textoBotonLogin');
  
  if (!boton || !textoBoton) return;
  
  const token = localStorage.getItem('authToken');
  const nombreUsuario = localStorage.getItem('nombre_usuario');
  
  if (token && token.trim() !== '') {
    // Usuario autenticado
    boton.className = "bg-green-500/20 hover:bg-green-500/30 backdrop-blur-sm text-white px-4 py-3 rounded-xl transition-all duration-300 hover:transform hover:scale-105 border border-green-400/20";
    textoBoton.textContent = nombreUsuario || 'Autenticado';
    boton.title = 'Usuario autenticado correctamente';
  } else {
    // Usuario no autenticado
    boton.className = "bg-amber-500/20 hover:bg-amber-500/30 backdrop-blur-sm text-white px-4 py-3 rounded-xl transition-all duration-300 hover:transform hover:scale-105 border border-amber-400/20";
    textoBoton.textContent = 'Login';
    boton.title = 'Hacer clic para ir a la página de login';
  }
}

/**
 * Verifica el estado de autenticación o redirige al login
 */
function verificarOIrALogin() {
  const token = localStorage.getItem('authToken');
  
  if (token && token.trim() !== '') {
    // Usuario ya autenticado, mostrar información
    const nombreUsuario = localStorage.getItem('nombre_usuario') || 'Usuario';
    const rol = localStorage.getItem('rol') || 'Sin rol';
    const empresa = localStorage.getItem('id_empresa') || 'Sin empresa';
    
    Swal.fire({
      icon: 'info',
      title: 'Estado de Autenticación',
      html: `
        <div class="text-left">
          <p><strong>Usuario:</strong> ${nombreUsuario}</p>
          <p><strong>Rol:</strong> ${rol}</p>
          <p><strong>Empresa ID:</strong> ${empresa}</p>
          <p><strong>Token:</strong> Válido ✅</p>
        </div>
      `,
      showCancelButton: true,
      confirmButtonText: 'Cerrar Sesión',
      cancelButtonText: 'Mantener Sesión',
      confirmButtonColor: '#ef4444'
    }).then((result) => {
      if (result.isConfirmed) {
        cerrarSesion();
      }
    });
  } else {
    // Usuario no autenticado, ir al login
    window.location.href = '../signin.html';
  }
}

/**
 * Cierra la sesión del usuario
 */
function cerrarSesion() {
  localStorage.removeItem('authToken');
  localStorage.removeItem('id_empresa');
  localStorage.removeItem('id_usuario');
  localStorage.removeItem('rol');
  localStorage.removeItem('nombre_usuario');
  localStorage.removeItem('datosUsuario');
  
  Swal.fire({
    icon: 'success',
    title: 'Sesión Cerrada',
    text: 'Tu sesión ha sido cerrada exitosamente',
    timer: 1500,
    showConfirmButton: false
  }).then(() => {
    window.location.href = '../signin.html';
  });
}

// ====================================================================
// FUNCIONES PÚBLICAS PARA INTEGRACIÓN CON OTROS MÓDULOS
// ====================================================================

/**
 * Configura el formulario de agregar moto para uso externo (ej: consignaciones)
 * Esta función puede ser llamada desde otros módulos para reutilizar la funcionalidad
 */
window.configurarFormularioAgregarMoto = function() {
  console.log('🏍️ Configurando formulario de moto para integración externa...');
  
  // No ejecutar la inicialización automática si ya se ejecutó
  if (window.agregarMotoInicializado) {
    console.log('✅ Formulario ya inicializado previamente');
    return;
  }
  
  // Marcar como inicializado
  window.agregarMotoInicializado = true;
  
  // Configurar tema inicial
  if (typeof configurarTema === 'function') {
    configurarTema();
  }
  
  // Cargar datos del usuario
  if (typeof cargarDatosUsuario === 'function') {
    cargarDatosUsuario();
  }
  
  // Cargar marcas disponibles
  if (typeof cargarMarcas === 'function') {
    cargarMarcas();
  }
  
  // Configurar fecha por defecto
  if (typeof configurarFechaDefecto === 'function') {
    configurarFechaDefecto();
  }
  
  // Configurar eventos del formulario
  if (typeof configurarEventosFormulario === 'function') {
    configurarEventosFormulario();
  }
  
  // Inicializar wizard dinámico
  if (typeof inicializarWizardDinamico === 'function') {
    inicializarWizardDinamico();
  }
  
  console.log('✅ Formulario de moto configurado para integración externa');
};

/**
 * Obtiene los datos actuales del formulario de moto
 */
window.obtenerDatosMoto = function() {
  const datos = {
    // Información básica
    marca: document.getElementById('marca_moto')?.value || '',
    modelo: document.getElementById('modelo')?.value || '',
    año: document.getElementById('año')?.value || '',
    color: document.getElementById('color')?.value || '',
    kilometraje: document.getElementById('kilometraje')?.value || '',
    precio: document.getElementById('precio')?.value || '',
    
    // Especificaciones técnicas
    cilindrada: document.getElementById('cilindrada')?.value || '',
    tipo_motor: document.getElementById('tipo_motor')?.value || '',
    transmision: document.getElementById('transmision')?.value || '',
    combustible: document.getElementById('combustible')?.value || '',
    
    // Estado comercial
    estado: document.getElementById('estado')?.value || '',
    disponibilidad: document.getElementById('disponibilidad')?.value || '',
    observaciones: document.getElementById('observaciones')?.value || '',
    
    // Información adicional si está disponible
    dueno_anterior: document.getElementById('dueno_anterior')?.value || '',
    numero_serie: document.getElementById('numero_serie')?.value || '',
    patente: document.getElementById('patente')?.value || ''
  };
  
  return datos;
};

/**
 * Valida que el formulario de moto esté completo
 */
window.validarFormularioAgregarMoto = function() {
  const camposRequeridos = [
    { id: 'marca_moto', nombre: 'Marca' },
    { id: 'modelo', nombre: 'Modelo' },
    { id: 'año', nombre: 'Año' },
    { id: 'precio', nombre: 'Precio' }
  ];
  
  for (const campo of camposRequeridos) {
    const elemento = document.getElementById(campo.id);
    if (!elemento || !elemento.value.trim()) {
      Swal.fire({
        icon: 'warning',
        title: 'Campo requerido',
        text: `El campo ${campo.nombre} es obligatorio.`
      });
      
      // Enfocar el campo faltante
      if (elemento) {
        elemento.focus();
      }
      
      return false;
    }
  }
  
  return true;
};

/**
 * Limpia el formulario de moto
 */
window.limpiarFormularioMoto = function() {
  const formulario = document.getElementById('form-agregar-moto');
  if (formulario) {
    formulario.reset();
  }
  
  // Limpiar también datos globales
  datosCompletos = {};
  documentosSubidos = [];
  
  console.log('✅ Formulario de moto limpiado');
};
